"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPureDatasetQualityDetailsControllerStateMachine = exports.createDatasetQualityDetailsControllerStateMachine = void 0;
var _xstate = require("xstate");
var _timerange = require("@kbn/timerange");
var _utils = require("../../../common/utils");
var _notifications = require("../common/notifications");
var _notifications2 = require("./notifications");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPureDatasetQualityDetailsControllerStateMachine = initialContext => (0, _xstate.createMachine)({
  id: 'DatasetQualityDetailsController',
  context: initialContext,
  predictableActionArguments: true,
  initial: 'initializing',
  states: {
    initializing: {
      type: 'parallel',
      states: {
        nonAggregatableDataset: {
          initial: 'fetching',
          states: {
            fetching: {
              invoke: {
                src: 'checkDatasetIsAggregatable',
                onDone: {
                  target: 'done',
                  actions: ['storeDatasetAggregatableStatus']
                },
                onError: [{
                  target: '#DatasetQualityDetailsController.indexNotFound',
                  cond: 'isIndexNotFoundError'
                }, {
                  target: 'done',
                  actions: ['notifyFailedFetchForAggregatableDatasets']
                }]
              }
            },
            done: {
              on: {
                UPDATE_TIME_RANGE: {
                  target: 'fetching',
                  actions: ['storeTimeRange']
                }
              }
            }
          }
        },
        dataStreamDetails: {
          initial: 'fetching',
          states: {
            fetching: {
              invoke: {
                src: 'loadDataStreamDetails',
                onDone: {
                  target: 'done',
                  actions: ['storeDataStreamDetails']
                },
                onError: [{
                  target: '#DatasetQualityDetailsController.indexNotFound',
                  cond: 'isIndexNotFoundError'
                }, {
                  target: 'done',
                  actions: ['notifyFetchDataStreamDetailsFailed']
                }]
              }
            },
            done: {
              on: {
                UPDATE_TIME_RANGE: {
                  target: 'fetching',
                  actions: ['storeTimeRange']
                },
                BREAKDOWN_FIELD_CHANGE: {
                  target: '#DatasetQualityDetailsController.initializing.checkBreakdownFieldIsEcs.fetching',
                  actions: ['storeBreakDownField']
                }
              }
            }
          }
        },
        checkBreakdownFieldIsEcs: {
          initial: 'fetching',
          states: {
            fetching: {
              invoke: {
                src: 'checkBreakdownFieldIsEcs',
                onDone: {
                  target: 'done',
                  actions: ['storeBreakdownFieldEcsStatus']
                },
                onError: {
                  target: 'done',
                  actions: ['notifyCheckBreakdownFieldIsEcsFailed']
                }
              }
            },
            done: {}
          }
        },
        dataStreamSettings: {
          initial: 'fetchingDataStreamSettings',
          states: {
            fetchingDataStreamSettings: {
              invoke: {
                src: 'loadDataStreamSettings',
                onDone: {
                  target: 'loadingIntegrationsAndDegradedFields',
                  actions: ['storeDataStreamSettings']
                },
                onError: [{
                  target: '#DatasetQualityDetailsController.indexNotFound',
                  cond: 'isIndexNotFoundError'
                }, {
                  target: 'done',
                  actions: ['notifyFetchDataStreamSettingsFailed']
                }]
              }
            },
            loadingIntegrationsAndDegradedFields: {
              type: 'parallel',
              states: {
                dataStreamDegradedFields: {
                  initial: 'fetching',
                  states: {
                    fetching: {
                      invoke: {
                        src: 'loadDegradedFields',
                        onDone: {
                          target: 'done',
                          actions: ['storeDegradedFields', 'raiseDegradedFieldsLoaded']
                        },
                        onError: [{
                          target: '#DatasetQualityDetailsController.indexNotFound',
                          cond: 'isIndexNotFoundError'
                        }, {
                          target: 'done'
                        }]
                      }
                    },
                    done: {
                      on: {
                        UPDATE_TIME_RANGE: {
                          target: 'fetching',
                          actions: ['resetDegradedFieldPageAndRowsPerPage']
                        },
                        UPDATE_DEGRADED_FIELDS_TABLE_CRITERIA: {
                          target: 'done',
                          actions: ['storeDegradedFieldTableOptions']
                        },
                        OPEN_DEGRADED_FIELD_FLYOUT: {
                          target: '#DatasetQualityDetailsController.initializing.degradedFieldFlyout.open',
                          actions: ['storeExpandedDegradedField']
                        },
                        TOGGLE_CURRENT_QUALITY_ISSUES: {
                          target: 'fetching',
                          actions: ['toggleCurrentQualityIssues']
                        }
                      }
                    }
                  }
                },
                integrationDetails: {
                  initial: 'fetching',
                  states: {
                    fetching: {
                      invoke: {
                        src: 'loadDataStreamIntegration',
                        onDone: {
                          target: 'done',
                          actions: ['storeDataStreamIntegration']
                        },
                        onError: {
                          target: 'done',
                          actions: ['notifyFetchDatasetIntegrationsFailed']
                        }
                      }
                    },
                    done: {}
                  }
                },
                integrationDashboards: {
                  initial: 'fetching',
                  states: {
                    fetching: {
                      invoke: {
                        src: 'loadIntegrationDashboards',
                        onDone: {
                          target: 'done',
                          actions: ['storeIntegrationDashboards']
                        },
                        onError: [{
                          target: 'unauthorized',
                          cond: 'checkIfActionForbidden'
                        }, {
                          target: 'done',
                          actions: ['notifyFetchIntegrationDashboardsFailed']
                        }]
                      }
                    },
                    done: {},
                    unauthorized: {
                      type: 'final'
                    }
                  }
                }
              },
              onDone: {
                target: 'done'
              }
            },
            done: {}
          },
          on: {
            UPDATE_TIME_RANGE: {
              target: '.fetchingDataStreamSettings'
            }
          }
        },
        degradedFieldFlyout: {
          initial: 'pending',
          states: {
            pending: {
              always: [{
                target: 'closed',
                cond: 'hasNoDegradedFieldsSelected'
              }]
            },
            open: {
              type: 'parallel',
              states: {
                ignoredValues: {
                  initial: 'fetching',
                  states: {
                    fetching: {
                      invoke: {
                        src: 'loadDegradedFieldValues',
                        onDone: {
                          target: 'done',
                          actions: ['storeDegradedFieldValues']
                        },
                        onError: [{
                          target: '#DatasetQualityDetailsController.indexNotFound',
                          cond: 'isIndexNotFoundError'
                        }, {
                          target: 'done'
                        }]
                      }
                    },
                    done: {}
                  }
                },
                analyze: {
                  initial: 'fetching',
                  states: {
                    fetching: {
                      invoke: {
                        src: 'analyzeDegradedField',
                        onDone: {
                          target: 'done',
                          actions: ['storeDegradedFieldAnalysis']
                        },
                        onError: {
                          target: 'done'
                        }
                      }
                    },
                    done: {}
                  }
                }
              },
              on: {
                CLOSE_DEGRADED_FIELD_FLYOUT: {
                  target: 'closed',
                  actions: ['storeExpandedDegradedField']
                },
                UPDATE_TIME_RANGE: {
                  target: '#DatasetQualityDetailsController.initializing.degradedFieldFlyout.open'
                }
              }
            },
            closed: {
              on: {
                OPEN_DEGRADED_FIELD_FLYOUT: {
                  target: '#DatasetQualityDetailsController.initializing.degradedFieldFlyout.open',
                  actions: ['storeExpandedDegradedField']
                }
              }
            }
          },
          on: {
            DEGRADED_FIELDS_LOADED: [{
              target: '.open',
              cond: 'shouldOpenFlyout'
            }, {
              target: '.closed',
              actions: ['storeExpandedDegradedField']
            }]
          }
        }
      }
    },
    indexNotFound: {
      entry: 'handleIndexNotFoundError'
    }
  }
}, {
  actions: {
    storeDatasetAggregatableStatus: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        isNonAggregatable: !event.data.aggregatable
      } : {};
    }),
    storeTimeRange: (0, _xstate.assign)((context, event) => {
      return {
        timeRange: 'timeRange' in event ? event.timeRange : context.timeRange
      };
    }),
    storeDataStreamDetails: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        dataStreamDetails: event.data
      } : {};
    }),
    storeBreakDownField: (0, _xstate.assign)((_context, event) => {
      return 'breakdownField' in event ? {
        breakdownField: event.breakdownField
      } : {};
    }),
    storeBreakdownFieldEcsStatus: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        isBreakdownFieldEcs: event.data
      } : {};
    }),
    storeDegradedFields: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        degradedFields: {
          ...context.degradedFields,
          data: event.data.degradedFields
        }
      } : {};
    }),
    storeDegradedFieldValues: (0, _xstate.assign)((_, event) => {
      return 'data' in event ? {
        degradedFieldValues: event.data
      } : {};
    }),
    storeDegradedFieldAnalysis: (0, _xstate.assign)((_, event) => {
      return 'data' in event ? {
        degradedFieldAnalysis: event.data
      } : {};
    }),
    storeDegradedFieldTableOptions: (0, _xstate.assign)((context, event) => {
      return 'degraded_field_criteria' in event ? {
        degradedFields: {
          ...context.degradedFields,
          table: event.degraded_field_criteria
        }
      } : {};
    }),
    storeExpandedDegradedField: (0, _xstate.assign)((_, event) => {
      return {
        expandedDegradedField: 'fieldName' in event ? event.fieldName : undefined
      };
    }),
    toggleCurrentQualityIssues: (0, _xstate.assign)(context => {
      return {
        showCurrentQualityIssues: !context.showCurrentQualityIssues
      };
    }),
    raiseDegradedFieldsLoaded: (0, _xstate.raise)('DEGRADED_FIELDS_LOADED'),
    resetDegradedFieldPageAndRowsPerPage: (0, _xstate.assign)((context, _event) => ({
      degradedFields: {
        ...context.degradedFields,
        table: {
          ...context.degradedFields.table,
          page: 0,
          rowsPerPage: 10
        }
      }
    })),
    storeDataStreamSettings: (0, _xstate.assign)((_context, event) => {
      return 'data' in event ? {
        dataStreamSettings: event.data
      } : {};
    }),
    storeDataStreamIntegration: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        integration: event.data
      } : {};
    }),
    storeIntegrationDashboards: (0, _xstate.assign)((context, event) => {
      return 'data' in event ? {
        integrationDashboards: event.data
      } : {};
    }),
    handleIndexNotFoundError: (0, _xstate.assign)(() => {
      return {
        isIndexNotFoundError: true
      };
    })
  },
  guards: {
    checkIfActionForbidden: (context, event) => {
      return 'data' in event && typeof event.data === 'object' && 'statusCode' in event.data && event.data.statusCode === 403;
    },
    isIndexNotFoundError: (_, event) => {
      var _ref, _event$data$originalM;
      return (_ref = 'data' in event && typeof event.data === 'object' && 'statusCode' in event.data && event.data.statusCode === 500 && 'originalMessage' in event.data && ((_event$data$originalM = event.data.originalMessage) === null || _event$data$originalM === void 0 ? void 0 : _event$data$originalM.includes('index_not_found_exception'))) !== null && _ref !== void 0 ? _ref : false;
    },
    shouldOpenFlyout: context => {
      var _context$degradedFiel;
      return Boolean(context.expandedDegradedField) && Boolean((_context$degradedFiel = context.degradedFields.data) === null || _context$degradedFiel === void 0 ? void 0 : _context$degradedFiel.some(field => field.name === context.expandedDegradedField));
    },
    hasNoDegradedFieldsSelected: context => {
      return !Boolean(context.expandedDegradedField);
    }
  }
});
exports.createPureDatasetQualityDetailsControllerStateMachine = createPureDatasetQualityDetailsControllerStateMachine;
const createDatasetQualityDetailsControllerStateMachine = ({
  initialContext,
  plugins,
  toasts,
  dataStreamStatsClient,
  dataStreamDetailsClient
}) => createPureDatasetQualityDetailsControllerStateMachine(initialContext).withConfig({
  actions: {
    notifyFailedFetchForAggregatableDatasets: (_context, event) => (0, _notifications.fetchNonAggregatableDatasetsFailedNotifier)(toasts, event.data),
    notifyFetchDataStreamDetailsFailed: (_context, event) => (0, _notifications2.fetchDataStreamDetailsFailedNotifier)(toasts, event.data),
    notifyCheckBreakdownFieldIsEcsFailed: (_context, event) => (0, _notifications2.assertBreakdownFieldEcsFailedNotifier)(toasts, event.data),
    notifyFetchDataStreamSettingsFailed: (_context, event) => (0, _notifications2.fetchDataStreamSettingsFailedNotifier)(toasts, event.data),
    notifyFetchIntegrationDashboardsFailed: (_context, event) => (0, _notifications2.fetchIntegrationDashboardsFailedNotifier)(toasts, event.data),
    notifyFetchDatasetIntegrationsFailed: (context, event) => {
      var _context$dataStreamSe;
      const integrationName = 'dataStreamSettings' in context ? (_context$dataStreamSe = context.dataStreamSettings) === null || _context$dataStreamSe === void 0 ? void 0 : _context$dataStreamSe.integration : undefined;
      return (0, _notifications2.fetchDataStreamIntegrationFailedNotifier)(toasts, event.data, integrationName);
    }
  },
  services: {
    checkDatasetIsAggregatable: context => {
      const {
        type
      } = (0, _utils.indexNameToDataStreamParts)(context.dataStream);
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      return dataStreamStatsClient.getNonAggregatableDatasets({
        types: [type],
        start,
        end,
        dataStream: context.dataStream
      });
    },
    loadDataStreamDetails: context => {
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      return dataStreamDetailsClient.getDataStreamDetails({
        dataStream: context.dataStream,
        start,
        end
      });
    },
    checkBreakdownFieldIsEcs: async context => {
      if (context.breakdownField) {
        var _fields$context$break;
        const allowedFieldSources = ['ecs', 'metadata'];

        // This timeout is to avoid a runtime error that randomly happens on breakdown field change
        // TypeError: Cannot read properties of undefined (reading 'timeFieldName')
        await new Promise(res => setTimeout(res, 300));
        const client = await plugins.fieldsMetadata.getClient();
        const {
          fields
        } = await client.find({
          attributes: ['source'],
          fieldNames: [context.breakdownField]
        });
        const breakdownFieldSource = (_fields$context$break = fields[context.breakdownField]) === null || _fields$context$break === void 0 ? void 0 : _fields$context$break.source;
        return !!(breakdownFieldSource && allowedFieldSources.includes(breakdownFieldSource));
      }
      return false;
    },
    loadDegradedFields: context => {
      const {
        startDate: start,
        endDate: end
      } = (0, _timerange.getDateISORange)(context.timeRange);
      if (!(context !== null && context !== void 0 && context.isNonAggregatable)) {
        return dataStreamDetailsClient.getDataStreamDegradedFields({
          dataStream: context.showCurrentQualityIssues && 'dataStreamSettings' in context && context.dataStreamSettings ? context.dataStreamSettings.lastBackingIndexName : context.dataStream,
          start,
          end
        });
      }
      return Promise.resolve();
    },
    loadDegradedFieldValues: context => {
      if ('expandedDegradedField' in context && context.expandedDegradedField) {
        return dataStreamDetailsClient.getDataStreamDegradedFieldValues({
          dataStream: context.dataStream,
          degradedField: context.expandedDegradedField
        });
      }
      return Promise.resolve();
    },
    analyzeDegradedField: context => {
      var _context$degradedFiel2, _context$degradedFiel3;
      if (context !== null && context !== void 0 && (_context$degradedFiel2 = context.degradedFields) !== null && _context$degradedFiel2 !== void 0 && (_context$degradedFiel3 = _context$degradedFiel2.data) !== null && _context$degradedFiel3 !== void 0 && _context$degradedFiel3.length) {
        const selectedDegradedField = context.degradedFields.data.find(field => field.name === context.expandedDegradedField);
        if (selectedDegradedField) {
          return dataStreamDetailsClient.analyzeDegradedField({
            dataStream: context.dataStream,
            degradedField: context.expandedDegradedField,
            lastBackingIndex: selectedDegradedField.indexFieldWasLastPresentIn
          });
        }
      }
      return Promise.resolve();
    },
    loadDataStreamSettings: context => {
      return dataStreamDetailsClient.getDataStreamSettings({
        dataStream: context.dataStream
      });
    },
    loadDataStreamIntegration: context => {
      var _context$dataStreamSe2;
      if ('dataStreamSettings' in context && (_context$dataStreamSe2 = context.dataStreamSettings) !== null && _context$dataStreamSe2 !== void 0 && _context$dataStreamSe2.integration) {
        return dataStreamDetailsClient.getDataStreamIntegration({
          integrationName: context.dataStreamSettings.integration
        });
      }
      return Promise.resolve();
    },
    loadIntegrationDashboards: context => {
      var _context$dataStreamSe3;
      if ('dataStreamSettings' in context && (_context$dataStreamSe3 = context.dataStreamSettings) !== null && _context$dataStreamSe3 !== void 0 && _context$dataStreamSe3.integration) {
        return dataStreamDetailsClient.getIntegrationDashboards({
          integration: context.dataStreamSettings.integration
        });
      }
      return Promise.resolve();
    }
  }
});
exports.createDatasetQualityDetailsControllerStateMachine = createDatasetQualityDetailsControllerStateMachine;