"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataStreamDetailsClient = void 0;
var _ioTsUtils = require("@kbn/io-ts-utils");
var _api_types = require("../../../common/api_types");
var _integration = require("../../../common/data_streams_stats/integration");
var _errors = require("../../../common/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DataStreamDetailsClient {
  constructor(http) {
    this.http = http;
  }
  async getDataStreamSettings({
    dataStream
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/settings`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream settings": ${error}`, error);
    });
    const dataStreamSettings = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamsSettingsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data stream settings response: ${message}"`))(response);
    return dataStreamSettings;
  }
  async getDataStreamDetails({
    dataStream,
    start,
    end
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/details`, {
      query: {
        start,
        end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream details": ${error}`, error);
    });
    const dataStreamDetails = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamsDetailsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data stream details response: ${message}"`))(response);
    return dataStreamDetails;
  }
  async getDataStreamDegradedFields({
    dataStream,
    start,
    end
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/degraded_fields`, {
      query: {
        start,
        end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream degraded fields": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamDegradedFieldsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data stream degraded fields response: ${message}"`))(response);
  }
  async getDataStreamDegradedFieldValues({
    dataStream,
    degradedField
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/degraded_field/${degradedField}/values`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream degraded field Value": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.degradedFieldValuesRt, message => new _errors.DatasetQualityError(`Failed to decode data stream degraded field values response: ${message}"`))(response);
  }
  async getIntegrationDashboards({
    integration
  }) {
    const response = await this.http.get(`/internal/dataset_quality/integrations/${integration}/dashboards`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch integration dashboards": ${error}`, error);
    });
    const {
      dashboards
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.integrationDashboardsRT, message => new _errors.DatasetQualityError(`Failed to decode integration dashboards response: ${message}"`))(response);
    return dashboards;
  }
  async getDataStreamIntegration(params) {
    const {
      integrationName
    } = params;
    const response = await this.http.get('/internal/dataset_quality/integrations').catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch integrations: ${error}`, error);
    });
    const {
      integrations
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.getIntegrationsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode integrations response: ${message}`))(response);
    const integration = integrations.find(i => i.name === integrationName);
    if (integration) return _integration.Integration.create(integration);
  }
  async analyzeDegradedField({
    dataStream,
    degradedField,
    lastBackingIndex
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/degraded_field/${degradedField}/analyze`, {
      query: {
        lastBackingIndex
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to analyze degraded field: ${degradedField} for datastream: ${dataStream}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.degradedFieldAnalysisRt, message => new _errors.DatasetQualityError(`Failed to decode the analysis response: ${message}`))(response);
  }
}
exports.DataStreamDetailsClient = DataStreamDetailsClient;