"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDatasetDetailsTelemetry = useDatasetDetailsTelemetry;
var _react = require("react");
var _timerange = require("@kbn/timerange");
var _use_dataset_quality_details_state = require("./use_dataset_quality_details_state");
var _telemetry = require("../services/telemetry");
var _utils = require("../../common/utils");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useDatasetDetailsTelemetry() {
  const {
    telemetryClient,
    datasetDetails,
    dataStreamDetails,
    timeRange,
    canUserViewIntegrations,
    canUserAccessDashboards,
    breakdownField,
    isNonAggregatable,
    isBreakdownFieldEcs,
    integrationDetails,
    loadingState
  } = (0, _use_dataset_quality_details_state.useDatasetQualityDetailsState)();
  const ebtProps = (0, _react.useMemo)(() => {
    if (dataStreamDetails && timeRange && !loadingState.dataStreamDetailsLoading) {
      return getDatasetDetailsEbtProps({
        datasetDetails,
        dataStreamDetails,
        timeRange,
        canUserViewIntegrations,
        canUserAccessDashboards,
        breakdownField,
        isNonAggregatable,
        isBreakdownFieldEcs,
        integration: integrationDetails.integration
      });
    }
    return undefined;
  }, [dataStreamDetails, timeRange, loadingState.dataStreamDetailsLoading, datasetDetails, canUserViewIntegrations, canUserAccessDashboards, breakdownField, isNonAggregatable, isBreakdownFieldEcs, integrationDetails.integration]);
  const startTracking = (0, _react.useCallback)(() => {
    telemetryClient.startDatasetDetailsTracking();
  }, [telemetryClient]);

  // Report opening dataset details
  (0, _react.useEffect)(() => {
    const datasetDetailsTrackingState = telemetryClient.getDatasetDetailsTrackingState();
    if (datasetDetailsTrackingState === 'started' && ebtProps) {
      telemetryClient.trackDatasetDetailsOpened(ebtProps);
    }
  }, [ebtProps, telemetryClient]);
  const trackDetailsNavigated = (0, _react.useCallback)((target, source, isDegraded = false) => {
    const datasetDetailsTrackingState = telemetryClient.getDatasetDetailsTrackingState();
    if ((datasetDetailsTrackingState === 'opened' || datasetDetailsTrackingState === 'navigated') && ebtProps) {
      telemetryClient.trackDatasetDetailsNavigated({
        ...ebtProps,
        filters: {
          is_degraded: isDegraded
        },
        target,
        source
      });
    } else {
      throw new Error('Cannot report dataset details navigation telemetry without required data and state');
    }
  }, [ebtProps, telemetryClient]);
  const trackDatasetDetailsBreakdownFieldChanged = (0, _react.useCallback)(() => {
    const datasetDetailsTrackingState = telemetryClient.getDatasetDetailsTrackingState();
    if ((datasetDetailsTrackingState === 'opened' || datasetDetailsTrackingState === 'navigated') && ebtProps) {
      telemetryClient.trackDatasetDetailsBreakdownFieldChanged({
        ...ebtProps,
        breakdown_field: ebtProps.breakdown_field
      });
    }
  }, [ebtProps, telemetryClient]);
  const wrapLinkPropsForTelemetry = (0, _react.useCallback)((props, target, source, isDegraded = false) => {
    return {
      ...props,
      onClick: event => {
        trackDetailsNavigated(target, source, isDegraded);
        if (props.onClick) {
          props.onClick(event);
        }
      }
    };
  }, [trackDetailsNavigated]);
  return {
    startTracking,
    trackDetailsNavigated,
    wrapLinkPropsForTelemetry,
    navigationTargets: _telemetry.NavigationTarget,
    navigationSources: _telemetry.NavigationSource,
    trackDatasetDetailsBreakdownFieldChanged
  };
}
function getDatasetDetailsEbtProps({
  datasetDetails,
  dataStreamDetails,
  timeRange,
  canUserViewIntegrations,
  canUserAccessDashboards,
  breakdownField,
  isNonAggregatable,
  isBreakdownFieldEcs,
  integration
}) {
  var _dataStreamDetails$de, _dataStreamDetails$do;
  const indexName = datasetDetails.rawName;
  const dataStream = {
    dataset: datasetDetails.name,
    namespace: datasetDetails.namespace,
    type: datasetDetails.type
  };
  const degradedDocs = (_dataStreamDetails$de = dataStreamDetails === null || dataStreamDetails === void 0 ? void 0 : dataStreamDetails.degradedDocsCount) !== null && _dataStreamDetails$de !== void 0 ? _dataStreamDetails$de : 0;
  const totalDocs = (_dataStreamDetails$do = dataStreamDetails === null || dataStreamDetails === void 0 ? void 0 : dataStreamDetails.docsCount) !== null && _dataStreamDetails$do !== void 0 ? _dataStreamDetails$do : 0;
  const degradedPercentage = totalDocs > 0 ? Number((degradedDocs / totalDocs * 100).toFixed(2)) : 0;
  const health = (0, _utils.mapPercentageToQuality)(degradedPercentage);
  const {
    startDate: from,
    endDate: to
  } = (0, _timerange.getDateISORange)(timeRange);
  return {
    index_name: indexName,
    data_stream: dataStream,
    privileges: {
      can_monitor_data_stream: true,
      can_view_integrations: canUserViewIntegrations,
      can_view_dashboards: canUserAccessDashboards
    },
    data_stream_aggregatable: !isNonAggregatable,
    data_stream_health: health,
    from,
    to,
    degraded_percentage: degradedPercentage,
    integration: integration === null || integration === void 0 ? void 0 : integration.name,
    breakdown_field: breakdownField ? isBreakdownFieldEcs === null ? _constants.UNKOWN_FIELD_PLACEHOLDER : getMaskedBreakdownField(breakdownField, isBreakdownFieldEcs) : breakdownField
  };
}
function getMaskedBreakdownField(breakdownField, isBreakdownFieldEcs) {
  return isBreakdownFieldEcs ? breakdownField : _constants.MASKED_FIELD_PLACEHOLDER;
}