"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCreateScriptedFieldRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _handle_errors = require("../util/handle_errors");
var _schemas = require("../../../schemas");
var _constants = require("../../../constants");
var _schema = require("../../schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerCreateScriptedFieldRoute = (router, getStartServices) => {
  router.versioned.post({
    path: '/api/index_patterns/index_pattern/{id}/scripted_field',
    access: 'public'
  }).addVersion({
    version: _constants.INITIAL_REST_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          })
        }, {
          unknowns: 'allow'
        }),
        body: _configSchema.schema.object({
          field: _schemas.fieldSpecSchema
        })
      },
      response: {
        200: {
          body: _schema.indexPatternsRuntimeResponseSchema
        }
      }
    }
  }, router.handleLegacyErrors((0, _handle_errors.handleErrors)(async (ctx, req, res) => {
    const core = await ctx.core;
    const savedObjectsClient = core.savedObjects.client;
    const elasticsearchClient = core.elasticsearch.client.asCurrentUser;
    const [,, {
      dataViewsServiceFactory
    }] = await getStartServices();
    const indexPatternsService = await dataViewsServiceFactory(savedObjectsClient, elasticsearchClient, req);
    const id = req.params.id;
    const {
      field
    } = req.body;
    if (!field.scripted) {
      throw new Error('Only scripted fields can be created.');
    }
    const indexPattern = await indexPatternsService.getDataViewLazy(id);
    if (await indexPattern.getFieldByName(field.name)) {
      throw new Error(`Field [name = ${field.name}] already exists.`);
    }
    indexPattern.upsertScriptedField({
      ...field,
      runtimeField: undefined,
      aggregatable: true,
      searchable: true
    });
    await indexPatternsService.updateSavedObject(indexPattern);
    const fieldObject = await indexPattern.getFieldByName(field.name);
    if (!fieldObject) throw new Error(`Could not create a field [name = ${field.name}].`);
    const body = {
      field: fieldObject.toSpec(),
      index_pattern: await indexPattern.toSpec({
        fieldParams: {
          fieldName: ['*']
        }
      })
    };
    return res.ok({
      headers: {
        'content-type': 'application/json'
      },
      body
    });
  })));
};
exports.registerCreateScriptedFieldRoute = registerCreateScriptedFieldRoute;