"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getIndices = getIndices;
exports.responseToItemArray = exports.getIndicesViaResolve = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const aliasLabel = _i18n.i18n.translate('dataViews.aliasLabel', {
  defaultMessage: 'Alias'
});
const dataStreamLabel = _i18n.i18n.translate('dataViews.dataStreamLabel', {
  defaultMessage: 'Data stream'
});
const indexLabel = _i18n.i18n.translate('dataViews.indexLabel', {
  defaultMessage: 'Index'
});
const frozenLabel = _i18n.i18n.translate('dataViews.frozenLabel', {
  defaultMessage: 'Frozen'
});
const rollupLabel = _i18n.i18n.translate('dataViews.rollupLabel', {
  defaultMessage: 'Rollup (deprecated)'
});
const getIndexTags = isRollupIndex => indexName => isRollupIndex(indexName) ? [{
  key: _types.INDEX_PATTERN_TYPE.ROLLUP,
  name: rollupLabel,
  color: 'warning'
}] : [];
const getIndicesViaResolve = async ({
  http,
  pattern,
  showAllIndices,
  isRollupIndex
}) => {
  const encodedPattern = encodeURIComponent(pattern);
  return http.get(`/internal/index-pattern-management/resolve_index/${encodedPattern}`, {
    query: showAllIndices ? {
      expand_wildcards: 'all'
    } : undefined
  }).then(response => {
    if (!response) {
      return [];
    } else {
      return responseToItemArray(response, getIndexTags(isRollupIndex));
    }
  });
};
exports.getIndicesViaResolve = getIndicesViaResolve;
async function getIndices({
  http,
  pattern: rawPattern = '',
  showAllIndices = false,
  isRollupIndex
}) {
  const pattern = rawPattern.trim();

  // Searching for `*:` fails for CCS environments. The search request
  // is worthless anyways as the we should only send a request
  // for a specific query (where we do not append *) if there is at
  // least a single character being searched for.
  if (pattern === '*:') {
    return [];
  }

  // This should never match anything so do not bother
  if (pattern === '') {
    return [];
  }

  // ES does not like just a `,*` and will throw a `[string_index_out_of_bounds_exception] String index out of range: 0`
  if (pattern.startsWith(',')) {
    return [];
  }
  return getIndicesViaResolve({
    http,
    pattern,
    showAllIndices,
    isRollupIndex
  }).catch(() => []);
}
const responseToItemArray = (response, getTags) => {
  const source = [];
  (response.indices || []).forEach(index => {
    const tags = [{
      key: 'index',
      name: indexLabel,
      color: 'default'
    }];
    const isFrozen = (index.attributes || []).includes(_types.ResolveIndexResponseItemIndexAttrs.FROZEN);
    tags.push(...getTags(index.name));
    if (isFrozen) {
      tags.push({
        name: frozenLabel,
        key: 'frozen',
        color: 'danger'
      });
    }
    source.push({
      name: index.name,
      tags,
      item: index
    });
  });
  (response.aliases || []).forEach(alias => {
    source.push({
      name: alias.name,
      tags: [{
        key: 'alias',
        name: aliasLabel,
        color: 'default'
      }],
      item: alias
    });
  });
  (response.data_streams || []).forEach(dataStream => {
    source.push({
      name: dataStream.name,
      tags: [{
        key: 'data_stream',
        name: dataStreamLabel,
        color: 'primary'
      }],
      item: dataStream
    });
  });
  return (0, _lodash.sortBy)(source, 'name');
};
exports.responseToItemArray = responseToItemArray;