"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsageMetricsHandler = void 0;
var _error_handler = require("../error_handler");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatStringParams = value => typeof value === 'string' ? [value] : value;
const getUsageMetricsHandler = dataUsageService => {
  const logger = dataUsageService.getLogger('usageMetricsRoute');
  return async (context, request, response) => {
    try {
      const core = await context.core;
      const esClient = core.elasticsearch.client.asCurrentUser;
      logger.debug(`Retrieving usage metrics`);
      const {
        from,
        to,
        metricTypes,
        dataStreams: requestDsNames
      } = request.body;

      // redundant check as we don't allow making requests via UI without data streams,
      // but it's here to make sure the request body is validated before requesting metrics from auto-ops
      if (!(requestDsNames !== null && requestDsNames !== void 0 && requestDsNames.length)) {
        return (0, _error_handler.errorHandler)(logger, response, new _utils.CustomHttpRequestError('[request body.dataStreams]: no data streams selected', 400));
      }
      const {
        data_streams: dataStreamsResponse
      } = await esClient.indices.getDataStream({
        name: requestDsNames,
        expand_wildcards: 'all'
      });
      const metrics = await dataUsageService.getMetrics({
        from,
        to,
        metricTypes: formatStringParams(metricTypes),
        dataStreams: formatStringParams(dataStreamsResponse.map(ds => ds.name))
      });
      const body = transformMetricsData(metrics);
      return response.ok({
        body
      });
    } catch (error) {
      logger.error(`Error retrieving usage metrics: ${error.message}`);
      return (0, _error_handler.errorHandler)(logger, response, error);
    }
  };
};
exports.getUsageMetricsHandler = getUsageMetricsHandler;
function transformMetricsData(data) {
  return {
    metrics: Object.fromEntries(Object.entries(data.metrics).map(([metricType, series]) => [metricType, series.map(metricSeries => ({
      name: metricSeries.name,
      data: metricSeries.data.map(([timestamp, value]) => ({
        x: timestamp,
        y: value
      }))
    }))]))
  };
}