"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataStreamAdapter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _create_or_update_component_template = require("./create_or_update_component_template");
var _create_or_update_data_stream = require("./create_or_update_data_stream");
var _create_or_update_index_template = require("./create_or_update_index_template");
var _install_with_timeout = require("./install_with_timeout");
var _resource_installer_utils = require("./resource_installer_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULT_FIELDS_LIMIT = 2500;
class DataStreamAdapter {
  constructor(name, options) {
    var _options$totalFieldsL;
    (0, _defineProperty2.default)(this, "kibanaVersion", void 0);
    (0, _defineProperty2.default)(this, "totalFieldsLimit", void 0);
    (0, _defineProperty2.default)(this, "componentTemplates", []);
    (0, _defineProperty2.default)(this, "indexTemplates", []);
    (0, _defineProperty2.default)(this, "installed", void 0);
    this.name = name;
    this.installed = false;
    this.kibanaVersion = options.kibanaVersion;
    this.totalFieldsLimit = (_options$totalFieldsL = options.totalFieldsLimit) !== null && _options$totalFieldsL !== void 0 ? _options$totalFieldsL : DEFAULT_FIELDS_LIMIT;
  }
  setComponentTemplate(params) {
    if (this.installed) {
      throw new Error('Cannot set component template after install');
    }
    this.componentTemplates.push((0, _resource_installer_utils.getComponentTemplate)(params));
  }
  setIndexTemplate(params) {
    if (this.installed) {
      throw new Error('Cannot set index template after install');
    }
    this.indexTemplates.push((0, _resource_installer_utils.getIndexTemplate)({
      ...params,
      indexPatterns: [this.name],
      kibanaVersion: this.kibanaVersion,
      totalFieldsLimit: this.totalFieldsLimit
    }));
  }
  getInstallFn({
    logger,
    pluginStop$,
    tasksTimeoutMs
  }) {
    return async (promise, description) => {
      try {
        await (0, _install_with_timeout.installWithTimeout)({
          installFn: () => promise,
          description,
          timeoutMs: tasksTimeoutMs,
          pluginStop$
        });
      } catch (err) {
        if (err instanceof _install_with_timeout.InstallShutdownError) {
          logger.info(err.message);
        } else {
          throw err;
        }
      }
    };
  }
  async install({
    logger,
    esClient: esClientToResolve,
    pluginStop$,
    tasksTimeoutMs
  }) {
    this.installed = true;
    const esClient = await esClientToResolve;
    const installFn = this.getInstallFn({
      logger,
      pluginStop$,
      tasksTimeoutMs
    });

    // Install component templates in parallel
    await Promise.all(this.componentTemplates.map(componentTemplate => installFn((0, _create_or_update_component_template.createOrUpdateComponentTemplate)({
      template: componentTemplate,
      esClient,
      logger,
      totalFieldsLimit: this.totalFieldsLimit
    }), `${componentTemplate.name} component template`)));

    // Install index templates in parallel
    await Promise.all(this.indexTemplates.map(indexTemplate => installFn((0, _create_or_update_index_template.createOrUpdateIndexTemplate)({
      template: indexTemplate,
      esClient,
      logger
    }), `${indexTemplate.name} index template`)));

    // create data stream when everything is ready
    await installFn((0, _create_or_update_data_stream.createOrUpdateDataStream)({
      name: this.name,
      esClient,
      logger,
      totalFieldsLimit: this.totalFieldsLimit
    }), `${this.name} data stream`);
  }
}
exports.DataStreamAdapter = DataStreamAdapter;