"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOrUpdateComponentTemplate = void 0;
var _std = require("@kbn/std");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const putIndexTemplateTotalFieldsLimitUsingComponentTemplate = async (esClient, componentTemplateName, totalFieldsLimit, logger) => {
  // Get all index templates and filter down to just the ones referencing this component template
  const {
    index_templates: indexTemplates
  } = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.getIndexTemplate(), {
    logger
  });
  const indexTemplatesUsingComponentTemplate = (indexTemplates !== null && indexTemplates !== void 0 ? indexTemplates : []).filter(indexTemplate => {
    var _indexTemplate$index_, _indexTemplate$index_2;
    return ((_indexTemplate$index_ = (_indexTemplate$index_2 = indexTemplate.index_template) === null || _indexTemplate$index_2 === void 0 ? void 0 : _indexTemplate$index_2.composed_of) !== null && _indexTemplate$index_ !== void 0 ? _indexTemplate$index_ : []).includes(componentTemplateName);
  });
  await (0, _std.asyncForEach)(indexTemplatesUsingComponentTemplate, async template => {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => {
      var _template$index_templ;
      return esClient.indices.putIndexTemplate({
        name: template.name,
        body: {
          ...template.index_template,
          template: {
            ...template.index_template.template,
            settings: {
              ...((_template$index_templ = template.index_template.template) === null || _template$index_templ === void 0 ? void 0 : _template$index_templ.settings),
              'index.mapping.total_fields.limit': totalFieldsLimit
            }
          }
        }
      });
    }, {
      logger
    });
  });
};
const createOrUpdateComponentTemplateHelper = async (esClient, template, totalFieldsLimit, logger) => {
  try {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.cluster.putComponentTemplate(template), {
      logger
    });
  } catch (error) {
    const limitErrorMatch = error.message.match(/Limit of total fields \[(\d+)\] has been exceeded/);
    if (limitErrorMatch != null) {
      // This error message occurs when there is an index template using this component template
      // that contains a field limit setting that using this component template exceeds
      // Specifically, this can happen for the ECS component template when we add new fields
      // to adhere to the ECS spec. Individual index templates specify field limits so if the
      // number of new ECS fields pushes the composed mapping above the limit, this error will
      // occur. We have to update the field limit inside the index template now otherwise we
      // can never update the component template

      logger.info(`Updating total_fields.limit from ${limitErrorMatch[1]} to ${totalFieldsLimit}`);
      await putIndexTemplateTotalFieldsLimitUsingComponentTemplate(esClient, template.name, totalFieldsLimit, logger);

      // Try to update the component template again
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.cluster.putComponentTemplate(template), {
        logger
      });
    } else {
      throw error;
    }
  }
};
const createOrUpdateComponentTemplate = async ({
  logger,
  esClient,
  template,
  totalFieldsLimit
}) => {
  logger.info(`Installing component template ${template.name}`);
  try {
    await createOrUpdateComponentTemplateHelper(esClient, template, totalFieldsLimit, logger);
  } catch (err) {
    logger.error(`Error installing component template ${template.name} - ${err.message}`);
    throw err;
  }
};
exports.createOrUpdateComponentTemplate = createOrUpdateComponentTemplate;