"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enhancedEsSearchStrategyProvider = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/kibana-utils-plugin/server");
var _report_search_error = require("../../report_search_error");
var _common = require("../../../../common");
var _request_utils = require("./request_utils");
var _response_utils = require("./response_utils");
var _search = require("../../collectors/search");
var _es_search = require("../es_search");
var _sanitize_request_params = require("../../sanitize_request_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const enhancedEsSearchStrategyProvider = (legacyConfig$, searchConfig, logger, usage, useInternalUser = false) => {
  function cancelAsyncSearch(id, {
    esClient
  }) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    return client.asyncSearch.delete({
      id
    });
  }
  async function asyncSearchStatus({
    id,
    ...request
  }, options, {
    esClient
  }) {
    var _request$params$keep_, _request$params;
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const keepAlive = (_request$params$keep_ = (_request$params = request.params) === null || _request$params === void 0 ? void 0 : _request$params.keep_alive) !== null && _request$params$keep_ !== void 0 ? _request$params$keep_ : (0, _request_utils.getDefaultAsyncGetParams)(searchConfig, options).keep_alive;
    const {
      body,
      headers
    } = await client.asyncSearch.status(
    // @ts-expect-error keep_alive was recently added and the types haven't been updated yet
    {
      id: id,
      keep_alive: keepAlive
    }, {
      ...options.transport,
      signal: options.abortSignal,
      meta: true
    });
    return (0, _response_utils.toAsyncKibanaSearchStatusResponse)(body, headers === null || headers === void 0 ? void 0 : headers.warning);
  }

  // Gets the current status of the async search request. If the request is complete, then queries for the results.
  async function getAsyncSearch({
    id,
    ...request
  }, options, {
    esClient
  }) {
    var _request$params2, _request$params3;
    if (!options.retrieveResults) {
      // First, request the status of the async search, and return the status if incomplete
      const status = await asyncSearchStatus({
        id,
        ...request
      }, options, {
        esClient
      });
      if ((0, _common.isRunningResponse)(status)) return status;
    }

    // Then, if the search is complete, request & return the final results
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const params = {
      ...(0, _request_utils.getDefaultAsyncGetParams)(searchConfig, options),
      ...((_request$params2 = request.params) !== null && _request$params2 !== void 0 && _request$params2.keep_alive ? {
        keep_alive: request.params.keep_alive
      } : {}),
      ...((_request$params3 = request.params) !== null && _request$params3 !== void 0 && _request$params3.wait_for_completion_timeout ? {
        wait_for_completion_timeout: request.params.wait_for_completion_timeout
      } : {})
    };
    const {
      body,
      headers
    } = await client.asyncSearch.get({
      ...params,
      id: id
    }, {
      ...options.transport,
      signal: options.abortSignal,
      meta: true
    });
    const response = (0, _es_search.shimHitsTotal)(body.response, options);
    return (0, _response_utils.toAsyncKibanaSearchResponse)({
      ...body,
      response
    }, headers === null || headers === void 0 ? void 0 : headers.warning);
  }
  async function submitAsyncSearch(request, options, {
    esClient,
    uiSettingsClient
  }) {
    var _meta$request;
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const params = {
      ...(await (0, _request_utils.getDefaultAsyncSubmitParams)(uiSettingsClient, searchConfig, options)),
      ...request.params
    };
    const {
      body,
      headers,
      meta
    } = await client.asyncSearch.submit(params, {
      ...options.transport,
      signal: options.abortSignal,
      meta: true
    });
    const response = (0, _es_search.shimHitsTotal)(body.response, options);
    return (0, _response_utils.toAsyncKibanaSearchResponse)({
      ...body,
      response
    }, headers === null || headers === void 0 ? void 0 : headers.warning, meta === null || meta === void 0 ? void 0 : (_meta$request = meta.request) === null || _meta$request === void 0 ? void 0 : _meta$request.params);
  }
  function asyncSearch({
    id,
    ...request
  }, options, deps) {
    const search = async () => {
      return id ? await getAsyncSearch({
        id,
        ...request
      }, options, deps) : await submitAsyncSearch(request, options, deps);
    };
    const cancel = async () => {
      if (!id || options.isStored) return;
      try {
        await cancelAsyncSearch(id, deps);
      } catch (e) {
        var _e$meta;
        // A 404 means either this search request does not exist, or that it is already cancelled
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) return;

        // Log all other (unexpected) error messages
        logger.error(`cancelAsyncSearch error: ${e.message}`);
      }
    };
    return (0, _common.pollSearch)(search, cancel, {
      pollInterval: searchConfig.asyncSearch.pollInterval,
      ...options
    }).pipe((0, _rxjs.tap)(response => id = response.id), (0, _rxjs.tap)((0, _search.searchUsageObserver)(logger, usage)), (0, _rxjs.catchError)(e => {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }));
  }
  async function rollupSearch(request, options, {
    esClient,
    uiSettingsClient
  }) {
    const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
    const legacyConfig = await (0, _rxjs.firstValueFrom)(legacyConfig$);
    const {
      body,
      index,
      ...params
    } = request.params;
    const method = 'POST';
    const path = encodeURI(`/${index}/_rollup_search`);
    const querystring = {
      ...(0, _es_search.getShardTimeout)(legacyConfig),
      ...(await (0, _request_utils.getIgnoreThrottled)(uiSettingsClient)),
      ...(await (0, _es_search.getDefaultSearchParams)(uiSettingsClient)),
      ...params
    };
    try {
      var _esResponse$meta, _esResponse$meta$requ, _esResponse$meta2, _esResponse$meta2$req;
      const esResponse = await client.transport.request({
        method,
        path,
        body,
        querystring
      }, {
        signal: options === null || options === void 0 ? void 0 : options.abortSignal,
        meta: true
      });
      const response = esResponse.body;
      return {
        rawResponse: (0, _es_search.shimHitsTotal)(response, options),
        ...((_esResponse$meta = esResponse.meta) !== null && _esResponse$meta !== void 0 && (_esResponse$meta$requ = _esResponse$meta.request) !== null && _esResponse$meta$requ !== void 0 && _esResponse$meta$requ.params ? {
          requestParams: (0, _sanitize_request_params.sanitizeRequestParams)((_esResponse$meta2 = esResponse.meta) === null || _esResponse$meta2 === void 0 ? void 0 : (_esResponse$meta2$req = _esResponse$meta2.request) === null || _esResponse$meta2$req === void 0 ? void 0 : _esResponse$meta2$req.params)
        } : {}),
        ...(0, _es_search.getTotalLoaded)(response)
      };
    } catch (e) {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }
  }
  return {
    /**
     * @param request
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Observable<IEsSearchResponse<any>>`
     * @throws `KbnSearchError`
     */
    search: (request, options, deps) => {
      logger.debug(() => `search ${JSON.stringify(request.params) || request.id}`);
      if (request.indexType === _common.DataViewType.ROLLUP && deps.rollupsEnabled) {
        return (0, _rxjs.from)(rollupSearch(request, options, deps));
      } else {
        return asyncSearch(request, options, deps);
      }
    },
    /**
     * @param id async search ID to cancel, as returned from _async_search API
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    cancel: async (id, options, deps) => {
      logger.debug(`cancel ${id}`);
      try {
        await cancelAsyncSearch(id, deps);
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    },
    /**
     *
     * @param id async search ID to extend, as returned from _async_search API
     * @param keepAlive
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    extend: async (id, keepAlive, options, {
      esClient
    }) => {
      logger.debug(`extend ${id} by ${keepAlive}`);
      try {
        const client = useInternalUser ? esClient.asInternalUser : esClient.asCurrentUser;
        await client.asyncSearch.get({
          id,
          keep_alive: keepAlive
        });
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    }
  };
};
exports.enhancedEsSearchStrategyProvider = enhancedEsSearchStrategyProvider;