"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.saveDashboardState = exports.convertTimeToUTCString = void 0;
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _public = require("@kbn/data-plugin/public");
var _esQuery = require("@kbn/es-query");
var _ = require("..");
var _common = require("../../../../common");
var _dashboard_panel_converters = require("../../../../common/lib/dashboard_panel_converters");
var _dashboard_constants = require("../../../dashboard_constants");
var _dashboard_container = require("../../../dashboard_container");
var _dashboard_container_strings = require("../../../dashboard_container/_dashboard_container_strings");
var _dashboard_backup_service = require("../../dashboard_backup_service");
var _kibana_services = require("../../kibana_services");
var _dashboard_versioning = require("./dashboard_versioning");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const convertTimeToUTCString = time => {
  if ((0, _moment.default)(time).isValid()) {
    return (0, _moment.default)(time).utc().format('YYYY-MM-DDTHH:mm:ss.SSS[Z]');
  } else {
    // If it's not a valid moment date, then it should be a string representing a relative time
    // like 'now' or 'now-15m'.
    return time;
  }
};
exports.convertTimeToUTCString = convertTimeToUTCString;
const saveDashboardState = async ({
  controlGroupReferences,
  lastSavedId,
  saveOptions,
  currentState,
  panelReferences
}) => {
  var _prefixedPanelReferen;
  const {
    search: dataSearchService,
    query: {
      timefilter: {
        timefilter
      }
    }
  } = _kibana_services.dataService;
  const dashboardContentManagementCache = (0, _.getDashboardContentManagementCache)();
  const {
    tags,
    query,
    title,
    filters,
    timeRestore,
    description,
    // Dashboard options
    useMargins,
    syncColors,
    syncCursor,
    syncTooltips,
    hidePanelTitles,
    controlGroupInput
  } = currentState;
  let {
    panels
  } = currentState;
  let prefixedPanelReferences = panelReferences;
  if (saveOptions.saveAsCopy) {
    const {
      panels: newPanels,
      references: newPanelReferences
    } = (0, _dashboard_panel_converters.generateNewPanelIds)(panels, panelReferences);
    panels = newPanels;
    prefixedPanelReferences = newPanelReferences;
    //
    // do not need to generate new ids for controls.
    // ControlGroup Component is keyed on dashboard id so changing dashboard id mounts new ControlGroup Component.
    //
  }

  /**
   * Stringify filters and query into search source JSON
   */
  const {
    searchSourceJSON,
    searchSourceReferences
  } = await (async () => {
    const searchSource = await dataSearchService.searchSource.create();
    searchSource.setField('filter',
    // save only unpinned filters
    filters.filter(filter => !(0, _esQuery.isFilterPinned)(filter)));
    searchSource.setField('query', query);
    const rawSearchSourceFields = searchSource.getSerializedFields();
    const [fields, references] = (0, _public.extractSearchSourceReferences)(rawSearchSourceFields);
    return {
      searchSourceReferences: references,
      searchSourceJSON: JSON.stringify(fields)
    };
  })();

  /**
   * Stringify options and panels
   */
  const optionsJSON = JSON.stringify({
    useMargins,
    syncColors,
    syncCursor,
    syncTooltips,
    hidePanelTitles
  });
  const panelsJSON = JSON.stringify((0, _common.convertPanelMapToSavedPanels)(panels, true));

  /**
   * Parse global time filter settings
   */
  const {
    from,
    to
  } = timefilter.getTime();
  const timeFrom = timeRestore ? convertTimeToUTCString(from) : undefined;
  const timeTo = timeRestore ? convertTimeToUTCString(to) : undefined;
  const refreshInterval = timeRestore ? (0, _lodash.pick)(timefilter.getRefreshInterval(), ['display', 'pause', 'section', 'value']) : undefined;
  const rawDashboardAttributes = {
    version: (0, _dashboard_versioning.convertDashboardVersionToNumber)(_dashboard_container.LATEST_DASHBOARD_CONTAINER_VERSION),
    controlGroupInput,
    kibanaSavedObjectMeta: {
      searchSourceJSON
    },
    description: description !== null && description !== void 0 ? description : '',
    refreshInterval,
    timeRestore,
    optionsJSON,
    panelsJSON,
    timeFrom,
    title,
    timeTo
  };

  /**
   * Extract references from raw attributes and tags into the references array.
   */
  const {
    attributes,
    references: dashboardReferences
  } = (0, _common.extractReferences)({
    attributes: rawDashboardAttributes,
    references: searchSourceReferences
  }, {
    embeddablePersistableStateService: _kibana_services.embeddableService
  });
  const savedObjectsTaggingApi = _kibana_services.savedObjectsTaggingService === null || _kibana_services.savedObjectsTaggingService === void 0 ? void 0 : _kibana_services.savedObjectsTaggingService.getTaggingApi();
  const references = savedObjectsTaggingApi !== null && savedObjectsTaggingApi !== void 0 && savedObjectsTaggingApi.ui.updateTagsReferences ? savedObjectsTaggingApi === null || savedObjectsTaggingApi === void 0 ? void 0 : savedObjectsTaggingApi.ui.updateTagsReferences(dashboardReferences, tags) : dashboardReferences;
  const allReferences = [...references, ...((_prefixedPanelReferen = prefixedPanelReferences) !== null && _prefixedPanelReferen !== void 0 ? _prefixedPanelReferen : []), ...(controlGroupReferences !== null && controlGroupReferences !== void 0 ? controlGroupReferences : [])];

  /**
   * Save the saved object using the content management
   */
  const idToSaveTo = saveOptions.saveAsCopy ? undefined : lastSavedId;
  try {
    const result = idToSaveTo ? await _kibana_services.contentManagementService.client.update({
      id: idToSaveTo,
      contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
      data: attributes,
      options: {
        references: allReferences,
        /** perform a "full" update instead, where the provided attributes will fully replace the existing ones */
        mergeAttributes: false
      }
    }) : await _kibana_services.contentManagementService.client.create({
      contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
      data: attributes,
      options: {
        references: allReferences
      }
    });
    const newId = result.item.id;
    if (newId) {
      _kibana_services.coreServices.notifications.toasts.addSuccess({
        title: _dashboard_container_strings.dashboardSaveToastStrings.getSuccessString(currentState.title),
        className: 'eui-textBreakWord',
        'data-test-subj': 'saveDashboardSuccess'
      });

      /**
       * If the dashboard id has been changed, redirect to the new ID to keep the url param in sync.
       */
      if (newId !== lastSavedId) {
        (0, _dashboard_backup_service.getDashboardBackupService)().clearState(lastSavedId);
        return {
          redirectRequired: true,
          id: newId,
          references: allReferences
        };
      } else {
        dashboardContentManagementCache.deleteDashboard(newId); // something changed in an existing dashboard, so delete it from the cache so that it can be re-fetched
      }
    }
    return {
      id: newId,
      references: allReferences
    };
  } catch (error) {
    _kibana_services.coreServices.notifications.toasts.addDanger({
      title: _dashboard_container_strings.dashboardSaveToastStrings.getFailureString(currentState.title, error.message),
      'data-test-subj': 'saveDashboardFailure'
    });
    return {
      error
    };
  }
};
exports.saveDashboardState = saveDashboardState;