"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.startQueryPerformanceTracking = startQueryPerformanceTracking;
var _rxjs = require("rxjs");
var _ebtTools = require("@kbn/ebt-tools");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _dashboard_constants = require("../../../../dashboard_constants");
var _kibana_services = require("../../../../services/kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

let isFirstDashboardLoadOfSession = true;
const loadTypesMapping = {
  sessionFirstLoad: 0,
  // on first time the SO is loaded
  dashboardFirstLoad: 1,
  // on initial load navigating into it
  dashboardSubsequentLoad: 2 // on filter-refresh
};
function startQueryPerformanceTracking(dashboard) {
  return dashboard.children$.pipe((0, _rxjs.skipWhile)(children => {
    // Don't track render-status when the dashboard is still adding embeddables.
    return Object.values(children).length !== dashboard.getPanelCount();
  }), (0, _rxjs.map)(children => {
    // Filter for embeddables which publish phase events
    const childPhaseEventTrackers = [];
    const values = Object.values(children);
    for (const child of values) {
      if ((0, _presentationPublishing.apiPublishesPhaseEvents)(child)) {
        childPhaseEventTrackers.push(child);
      }
    }
    return childPhaseEventTrackers;
  }), (0, _rxjs.switchMap)(children => {
    if (children.length === 0) {
      return (0, _rxjs.of)([]); // map to empty stream
    }
    // Map to new stream of phase-events for each embeddable
    return (0, _rxjs.combineLatest)(children.map(child => child.phase$));
  }), (0, _rxjs.map)(latestPhaseEvents => {
    // Map individual render-state of panels to global render-state.
    return latestPhaseEvents.some(phaseEvent => {
      return phaseEvent && phaseEvent.status !== 'rendered';
    });
  }), (0, _rxjs.startWith)(false), (0, _rxjs.pairwise)()).subscribe(([wasDashboardStillLoading, isDashboardStillLoading]) => {
    const panelCount = dashboard.getPanelCount();
    const now = performance.now();
    const loadType = isFirstDashboardLoadOfSession ? 'sessionFirstLoad' : dashboard.firstLoad ? 'dashboardFirstLoad' : 'dashboardSubsequentLoad';
    const queryHasStarted = !wasDashboardStillLoading && isDashboardStillLoading;
    const queryHasFinished = wasDashboardStillLoading && !isDashboardStillLoading;
    if (dashboard.firstLoad && (panelCount === 0 || queryHasFinished)) {
      /**
       * we consider the Dashboard creation to be finished when all the panels are loaded.
       */
      dashboard.creationEndTime = now;
      isFirstDashboardLoadOfSession = false;
      dashboard.firstLoad = false;
    }
    if (queryHasStarted) {
      dashboard.lastLoadStartTime = now;
      return;
    }
    if (queryHasFinished) {
      var _dashboard$lastLoadSt, _dashboard$creationEn, _dashboard$creationSt;
      const timeToData = now - ((_dashboard$lastLoadSt = dashboard.lastLoadStartTime) !== null && _dashboard$lastLoadSt !== void 0 ? _dashboard$lastLoadSt : now);
      const completeLoadDuration = ((_dashboard$creationEn = dashboard.creationEndTime) !== null && _dashboard$creationEn !== void 0 ? _dashboard$creationEn : now) - ((_dashboard$creationSt = dashboard.creationStartTime) !== null && _dashboard$creationSt !== void 0 ? _dashboard$creationSt : now);
      reportPerformanceMetrics({
        timeToData,
        panelCount,
        totalLoadTime: completeLoadDuration,
        loadType
      });
    }
  });
}
function reportPerformanceMetrics({
  timeToData,
  panelCount,
  totalLoadTime,
  loadType
}) {
  const duration = loadType === 'dashboardSubsequentLoad' ? timeToData : Math.max(timeToData, totalLoadTime);
  const e = {
    eventName: _dashboard_constants.DASHBOARD_LOADED_EVENT,
    duration,
    key1: 'time_to_data',
    value1: timeToData,
    key2: 'num_of_panels',
    value2: panelCount,
    key4: 'load_type',
    value4: loadTypesMapping[loadType]
  };
  (0, _ebtTools.reportPerformanceMetricEvent)(_kibana_services.coreServices.analytics, e);
}