"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeDashboard = exports.createDashboard = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _uuid = require("uuid");
var _contentManagementContentInsightsPublic = require("@kbn/content-management-content-insights-public");
var _public = require("@kbn/data-plugin/public");
var _public2 = require("@kbn/embeddable-plugin/public");
var _public3 = require("@kbn/presentation-util-plugin/public");
var _dashboard_constants = require("../../../dashboard_constants");
var _dashboard_backup_service = require("../../../services/dashboard_backup_service");
var _dashboard_content_management_service = require("../../../services/dashboard_content_management_service");
var _kibana_services = require("../../../services/kibana_services");
var _get_dashboard_capabilities = require("../../../utils/get_dashboard_capabilities");
var _place_new_panel_strategies = require("../../panel_placement/place_new_panel_strategies");
var _dashboard_diffing_integration = require("../../state/diffing/dashboard_diffing_integration");
var _dashboard_container = require("../dashboard_container");
var _sync_dashboard_data_views = require("./data_views/sync_dashboard_data_views");
var _query_performance_tracking = require("./performance/query_performance_tracking");
var _start_dashboard_search_session_integration = require("./search_sessions/start_dashboard_search_session_integration");
var _sync_dashboard_unified_search_state = require("./unified_search/sync_dashboard_unified_search_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Builds a new Dashboard from scratch.
 */
const createDashboard = async (creationOptions, dashboardCreationStartTime, savedObjectId) => {
  var _savedObjectResult$an, _creationOptions$isEm, _omit, _savedObjectResult$ma, _creationOptions$full;
  // --------------------------------------------------------------------------------------
  // Create method which allows work to be done on the dashboard container when it's ready.
  // --------------------------------------------------------------------------------------
  const dashboardContainerReady$ = new _rxjs.Subject();
  const untilDashboardReady = () => new Promise(resolve => {
    const subscription = dashboardContainerReady$.subscribe(container => {
      subscription.unsubscribe();
      resolve(container);
    });
  });

  // --------------------------------------------------------------------------------------
  // Lazy load required systems and Dashboard saved object.
  // --------------------------------------------------------------------------------------
  const reduxEmbeddablePackagePromise = (0, _public3.lazyLoadReduxToolsPackage)();
  const defaultDataViewExistsPromise = _kibana_services.dataService.dataViews.defaultDataViewExists();
  const dashboardContentManagementService = (0, _dashboard_content_management_service.getDashboardContentManagementService)();
  const dashboardSavedObjectPromise = dashboardContentManagementService.loadDashboardState({
    id: savedObjectId
  });
  const [reduxEmbeddablePackage, savedObjectResult] = await Promise.all([reduxEmbeddablePackagePromise, dashboardSavedObjectPromise, defaultDataViewExistsPromise /* the result is not used, but the side effect of setting the default data view is needed. */]);

  // --------------------------------------------------------------------------------------
  // Initialize Dashboard integrations
  // --------------------------------------------------------------------------------------
  const initializeResult = await initializeDashboard({
    loadDashboardReturn: savedObjectResult,
    untilDashboardReady,
    creationOptions
  });
  if (!initializeResult) return;
  const {
    input,
    searchSessionId
  } = initializeResult;

  // --------------------------------------------------------------------------------------
  // Build the dashboard container.
  // --------------------------------------------------------------------------------------
  const initialComponentState = {
    anyMigrationRun: (_savedObjectResult$an = savedObjectResult.anyMigrationRun) !== null && _savedObjectResult$an !== void 0 ? _savedObjectResult$an : false,
    isEmbeddedExternally: (_creationOptions$isEm = creationOptions === null || creationOptions === void 0 ? void 0 : creationOptions.isEmbeddedExternally) !== null && _creationOptions$isEm !== void 0 ? _creationOptions$isEm : false,
    lastSavedInput: (_omit = (0, _lodash.omit)(savedObjectResult === null || savedObjectResult === void 0 ? void 0 : savedObjectResult.dashboardInput, 'controlGroupInput')) !== null && _omit !== void 0 ? _omit : {
      ..._dashboard_constants.DEFAULT_DASHBOARD_INPUT,
      id: input.id
    },
    lastSavedId: savedObjectId,
    managed: (_savedObjectResult$ma = savedObjectResult.managed) !== null && _savedObjectResult$ma !== void 0 ? _savedObjectResult$ma : false,
    fullScreenMode: (_creationOptions$full = creationOptions === null || creationOptions === void 0 ? void 0 : creationOptions.fullScreenMode) !== null && _creationOptions$full !== void 0 ? _creationOptions$full : false
  };
  const dashboardContainer = new _dashboard_container.DashboardContainer(input, reduxEmbeddablePackage, searchSessionId, dashboardCreationStartTime, undefined, creationOptions, initialComponentState);

  // --------------------------------------------------------------------------------------
  // Start the diffing integration after all other integrations are set up.
  // --------------------------------------------------------------------------------------
  untilDashboardReady().then(container => {
    _dashboard_diffing_integration.startDiffingDashboardState.bind(container)(creationOptions);
  });
  dashboardContainerReady$.next(dashboardContainer);
  return dashboardContainer;
};

/**
 * Initializes a Dashboard and starts all of its integrations
 */
exports.createDashboard = createDashboard;
const initializeDashboard = async ({
  loadDashboardReturn,
  untilDashboardReady,
  creationOptions
}) => {
  var _dashboardBackupState, _loadDashboardReturn$, _creationOptions$getI;
  const {
    queryString,
    filterManager,
    timefilter: {
      timefilter: timefilterService
    }
  } = _kibana_services.dataService.query;
  const dashboardBackupService = (0, _dashboard_backup_service.getDashboardBackupService)();
  const {
    getInitialInput,
    searchSessionSettings,
    unifiedSearchSettings,
    validateLoadedSavedObject,
    useUnifiedSearchIntegration,
    useSessionStorageIntegration
  } = creationOptions !== null && creationOptions !== void 0 ? creationOptions : {};

  // --------------------------------------------------------------------------------------
  // Run validation.
  // --------------------------------------------------------------------------------------
  const validationResult = loadDashboardReturn && (validateLoadedSavedObject === null || validateLoadedSavedObject === void 0 ? void 0 : validateLoadedSavedObject(loadDashboardReturn));
  if (validationResult === 'invalid') {
    // throw error to stop the rest of Dashboard loading and make the factory return an ErrorEmbeddable.
    throw new Error('Dashboard failed saved object result validation');
  } else if (validationResult === 'redirected') {
    return;
  }

  // --------------------------------------------------------------------------------------
  // Combine input from saved object, and session storage
  // --------------------------------------------------------------------------------------
  const dashboardBackupState = dashboardBackupService.getState(loadDashboardReturn.dashboardId);
  const runtimePanelsToRestore = useSessionStorageIntegration ? (_dashboardBackupState = dashboardBackupState === null || dashboardBackupState === void 0 ? void 0 : dashboardBackupState.panels) !== null && _dashboardBackupState !== void 0 ? _dashboardBackupState : {} : {};
  const sessionStorageInput = (() => {
    if (!useSessionStorageIntegration) return;
    return dashboardBackupState === null || dashboardBackupState === void 0 ? void 0 : dashboardBackupState.dashboardState;
  })();
  const initialViewMode = (() => {
    if (loadDashboardReturn.managed || !(0, _get_dashboard_capabilities.getDashboardCapabilities)().showWriteControls) return _public2.ViewMode.VIEW;
    if (loadDashboardReturn.newDashboardCreated || dashboardBackupService.dashboardHasUnsavedEdits(loadDashboardReturn.dashboardId)) {
      return _public2.ViewMode.EDIT;
    }
    return dashboardBackupService.getViewMode();
  })();
  const combinedSessionInput = {
    ..._dashboard_constants.DEFAULT_DASHBOARD_INPUT,
    ...((_loadDashboardReturn$ = loadDashboardReturn === null || loadDashboardReturn === void 0 ? void 0 : loadDashboardReturn.dashboardInput) !== null && _loadDashboardReturn$ !== void 0 ? _loadDashboardReturn$ : {}),
    ...sessionStorageInput
  };

  // --------------------------------------------------------------------------------------
  // Combine input with overrides.
  // --------------------------------------------------------------------------------------
  const overrideInput = getInitialInput === null || getInitialInput === void 0 ? void 0 : getInitialInput();
  if (overrideInput !== null && overrideInput !== void 0 && overrideInput.panels) {
    /**
     * react embeddables and legacy embeddables share state very differently, so we need different
     * treatment here. TODO remove this distinction when we remove the legacy embeddable system.
     */
    const overridePanels = {};
    for (const panel of Object.values(overrideInput === null || overrideInput === void 0 ? void 0 : overrideInput.panels)) {
      if (_kibana_services.embeddableService.reactEmbeddableRegistryHasKey(panel.type)) {
        var _combinedSessionInput;
        overridePanels[panel.explicitInput.id] = {
          ...panel,
          /**
           * here we need to keep the state of the panel that was already in the Dashboard if one exists.
           * This is because this state will become the "last saved state" for this panel.
           */
          ...((_combinedSessionInput = combinedSessionInput.panels[panel.explicitInput.id]) !== null && _combinedSessionInput !== void 0 ? _combinedSessionInput : [])
        };
        /**
         * We also need to add the state of this react embeddable into the runtime state to be restored.
         */
        runtimePanelsToRestore[panel.explicitInput.id] = panel.explicitInput;
      } else {
        /**
         * if this is a legacy embeddable, the override state needs to completely overwrite the existing
         * state for this panel.
         */
        overridePanels[panel.explicitInput.id] = panel;
      }
    }

    /**
     * If this is a React embeddable, we leave the "panel" state as-is and add this state to the
     * runtime state to be restored on dashboard load.
     */
    overrideInput.panels = overridePanels;
  }
  const combinedOverrideInput = {
    ...combinedSessionInput,
    ...(initialViewMode ? {
      viewMode: initialViewMode
    } : {}),
    ...overrideInput
  };

  // --------------------------------------------------------------------------------------
  // Combine input from saved object, session storage, & passed input to create initial input.
  // --------------------------------------------------------------------------------------
  const initialDashboardInput = (0, _lodash.omit)((0, _lodash.cloneDeep)(combinedOverrideInput), 'controlGroupInput');

  // Back up any view mode passed in explicitly.
  if (overrideInput !== null && overrideInput !== void 0 && overrideInput.viewMode) {
    dashboardBackupService.storeViewMode(overrideInput === null || overrideInput === void 0 ? void 0 : overrideInput.viewMode);
  }
  initialDashboardInput.executionContext = {
    type: 'dashboard',
    description: initialDashboardInput.title
  };

  // --------------------------------------------------------------------------------------
  // Track references
  // --------------------------------------------------------------------------------------
  untilDashboardReady().then(dashboard => {
    var _loadDashboardReturn$2;
    dashboard.savedObjectReferences = loadDashboardReturn === null || loadDashboardReturn === void 0 ? void 0 : loadDashboardReturn.references;
    dashboard.controlGroupInput = loadDashboardReturn === null || loadDashboardReturn === void 0 ? void 0 : (_loadDashboardReturn$2 = loadDashboardReturn.dashboardInput) === null || _loadDashboardReturn$2 === void 0 ? void 0 : _loadDashboardReturn$2.controlGroupInput;
  });

  // --------------------------------------------------------------------------------------
  // Set up unified search integration.
  // --------------------------------------------------------------------------------------
  if (useUnifiedSearchIntegration && unifiedSearchSettings !== null && unifiedSearchSettings !== void 0 && unifiedSearchSettings.kbnUrlStateStorage) {
    const {
      query,
      filters,
      timeRestore,
      timeRange: savedTimeRange,
      refreshInterval: savedRefreshInterval
    } = initialDashboardInput;
    const {
      kbnUrlStateStorage
    } = unifiedSearchSettings;

    // apply filters and query to the query service
    filterManager.setAppFilters((0, _lodash.cloneDeep)(filters !== null && filters !== void 0 ? filters : []));
    queryString.setQuery(query !== null && query !== void 0 ? query : queryString.getDefaultQuery());

    /**
     * Get initial time range, and set up dashboard time restore if applicable
     */
    const initialTimeRange = (_kbnUrlStateStorage$g => {
      // if there is an explicit time range in the URL it always takes precedence.
      const urlOverrideTimeRange = (_kbnUrlStateStorage$g = kbnUrlStateStorage.get(_dashboard_constants.GLOBAL_STATE_STORAGE_KEY)) === null || _kbnUrlStateStorage$g === void 0 ? void 0 : _kbnUrlStateStorage$g.time;
      if (urlOverrideTimeRange) return urlOverrideTimeRange;

      // if this Dashboard has timeRestore return the time range that was saved with the dashboard.
      if (timeRestore && savedTimeRange) return savedTimeRange;

      // otherwise fall back to the time range from the timefilterService.
      return timefilterService.getTime();
    })();
    initialDashboardInput.timeRange = initialTimeRange;
    if (timeRestore) {
      if (savedTimeRange) timefilterService.setTime(savedTimeRange);
      if (savedRefreshInterval) timefilterService.setRefreshInterval(savedRefreshInterval);
    }

    // start syncing global query state with the URL.
    const {
      stop: stopSyncingQueryServiceStateWithUrl
    } = (0, _public.syncGlobalQueryStateWithUrl)(_kibana_services.dataService.query, kbnUrlStateStorage);
    untilDashboardReady().then(dashboardContainer => {
      const stopSyncingUnifiedSearchState = _sync_dashboard_unified_search_state.syncUnifiedSearchState.bind(dashboardContainer)(kbnUrlStateStorage);
      dashboardContainer.stopSyncingWithUnifiedSearch = () => {
        stopSyncingUnifiedSearchState();
        stopSyncingQueryServiceStateWithUrl();
      };
    });
  }

  // --------------------------------------------------------------------------------------
  // Place the incoming embeddable if there is one
  // --------------------------------------------------------------------------------------
  const incomingEmbeddable = creationOptions === null || creationOptions === void 0 ? void 0 : (_creationOptions$getI = creationOptions.getIncomingEmbeddable) === null || _creationOptions$getI === void 0 ? void 0 : _creationOptions$getI.call(creationOptions);
  if (incomingEmbeddable) {
    const scrolltoIncomingEmbeddable = (container, id) => {
      container.setScrollToPanelId(id);
      container.setHighlightPanelId(id);
    };
    initialDashboardInput.viewMode = _public2.ViewMode.EDIT; // view mode must always be edit to recieve an embeddable.
    if (incomingEmbeddable.embeddableId && Boolean(initialDashboardInput.panels[incomingEmbeddable.embeddableId])) {
      // this embeddable already exists, we will update the explicit input.
      const panelToUpdate = initialDashboardInput.panels[incomingEmbeddable.embeddableId];
      const sameType = panelToUpdate.type === incomingEmbeddable.type;
      panelToUpdate.type = incomingEmbeddable.type;
      const nextRuntimeState = {
        // if the incoming panel is the same type as what was there before we can safely spread the old panel's explicit input
        ...(sameType ? panelToUpdate.explicitInput : {}),
        ...incomingEmbeddable.input,
        id: incomingEmbeddable.embeddableId,
        // maintain hide panel titles setting.
        hidePanelTitles: panelToUpdate.explicitInput.hidePanelTitles
      };
      if (_kibana_services.embeddableService.reactEmbeddableRegistryHasKey(incomingEmbeddable.type)) {
        panelToUpdate.explicitInput = {
          id: panelToUpdate.explicitInput.id
        };
        runtimePanelsToRestore[incomingEmbeddable.embeddableId] = nextRuntimeState;
      } else {
        panelToUpdate.explicitInput = nextRuntimeState;
      }
      untilDashboardReady().then(container => scrolltoIncomingEmbeddable(container, incomingEmbeddable.embeddableId));
    } else {
      // otherwise this incoming embeddable is brand new and can be added after the dashboard container is created.

      untilDashboardReady().then(async container => {
        const createdEmbeddable = await (async _incomingEmbeddable$e => {
          // if there is no width or height we can add the panel using the default behaviour.
          if (!incomingEmbeddable.size) {
            return await container.addNewPanel({
              panelType: incomingEmbeddable.type,
              initialState: incomingEmbeddable.input
            });
          }

          // if the incoming embeddable has an explicit width or height we add the panel to the grid directly.
          const {
            width,
            height
          } = incomingEmbeddable.size;
          const currentPanels = container.getInput().panels;
          const embeddableId = (_incomingEmbeddable$e = incomingEmbeddable.embeddableId) !== null && _incomingEmbeddable$e !== void 0 ? _incomingEmbeddable$e : (0, _uuid.v4)();
          const {
            newPanelPlacement
          } = (0, _place_new_panel_strategies.runPanelPlacementStrategy)(_dashboard_constants.PanelPlacementStrategy.findTopLeftMostOpenSpace, {
            width: width !== null && width !== void 0 ? width : _dashboard_constants.DEFAULT_PANEL_WIDTH,
            height: height !== null && height !== void 0 ? height : _dashboard_constants.DEFAULT_PANEL_HEIGHT,
            currentPanels
          });
          const newPanelState = (() => {
            if (_kibana_services.embeddableService.reactEmbeddableRegistryHasKey(incomingEmbeddable.type)) {
              runtimePanelsToRestore[embeddableId] = incomingEmbeddable.input;
              return {
                explicitInput: {
                  id: embeddableId
                },
                type: incomingEmbeddable.type,
                gridData: {
                  ...newPanelPlacement,
                  i: embeddableId
                }
              };
            }
            return {
              explicitInput: {
                ...incomingEmbeddable.input,
                id: embeddableId
              },
              type: incomingEmbeddable.type,
              gridData: {
                ...newPanelPlacement,
                i: embeddableId
              }
            };
          })();
          container.updateInput({
            panels: {
              ...container.getInput().panels,
              [newPanelState.explicitInput.id]: newPanelState
            }
          });
          return await container.untilEmbeddableLoaded(embeddableId);
        })();
        if (createdEmbeddable) {
          scrolltoIncomingEmbeddable(container, createdEmbeddable.uuid);
        }
      });
    }
  }

  // --------------------------------------------------------------------------------------
  // Set restored runtime state for react embeddables.
  // --------------------------------------------------------------------------------------
  untilDashboardReady().then(dashboardContainer => {
    if (overrideInput !== null && overrideInput !== void 0 && overrideInput.controlGroupState) {
      dashboardContainer.setRuntimeStateForChild(_dashboard_backup_service.PANELS_CONTROL_GROUP_KEY, overrideInput.controlGroupState);
    }
    for (const idWithRuntimeState of Object.keys(runtimePanelsToRestore)) {
      const restoredRuntimeStateForChild = runtimePanelsToRestore[idWithRuntimeState];
      if (!restoredRuntimeStateForChild) continue;
      dashboardContainer.setRuntimeStateForChild(idWithRuntimeState, restoredRuntimeStateForChild);
    }
  });

  // --------------------------------------------------------------------------------------
  // Start the data views integration.
  // --------------------------------------------------------------------------------------
  untilDashboardReady().then(dashboardContainer => {
    dashboardContainer.integrationSubscriptions.add(_sync_dashboard_data_views.startSyncingDashboardDataViews.bind(dashboardContainer)());
  });

  // --------------------------------------------------------------------------------------
  // Start performance tracker
  // --------------------------------------------------------------------------------------
  untilDashboardReady().then(dashboardContainer => dashboardContainer.integrationSubscriptions.add((0, _query_performance_tracking.startQueryPerformanceTracking)(dashboardContainer)));

  // --------------------------------------------------------------------------------------
  // Start animating panel transforms 500 ms after dashboard is created.
  // --------------------------------------------------------------------------------------
  untilDashboardReady().then(dashboard => setTimeout(() => dashboard.setAnimatePanelTransforms(true), 500));

  // --------------------------------------------------------------------------------------
  // Set up search sessions integration.
  // --------------------------------------------------------------------------------------
  let initialSearchSessionId;
  if (searchSessionSettings) {
    const {
      sessionIdToRestore
    } = searchSessionSettings;

    // if this incoming embeddable has a session, continue it.
    if (incomingEmbeddable !== null && incomingEmbeddable !== void 0 && incomingEmbeddable.searchSessionId) {
      _kibana_services.dataService.search.session.continue(incomingEmbeddable.searchSessionId);
    }
    if (sessionIdToRestore) {
      _kibana_services.dataService.search.session.restore(sessionIdToRestore);
    }
    const existingSession = _kibana_services.dataService.search.session.getSessionId();
    initialSearchSessionId = sessionIdToRestore !== null && sessionIdToRestore !== void 0 ? sessionIdToRestore : existingSession && incomingEmbeddable ? existingSession : _kibana_services.dataService.search.session.start();
    untilDashboardReady().then(async container => {
      await container.untilContainerInitialized();
      _start_dashboard_search_session_integration.startDashboardSearchSessionIntegration.bind(container)(creationOptions === null || creationOptions === void 0 ? void 0 : creationOptions.searchSessionSettings);
    });
  }
  if (loadDashboardReturn.dashboardId && !incomingEmbeddable) {
    // We count a new view every time a user opens a dashboard, both in view or edit mode
    // We don't count views when a user is editing a dashboard and is returning from an editor after saving
    // however, there is an edge case that we now count a new view when a user is editing a dashboard and is returning from an editor by canceling
    // TODO: this should be revisited by making embeddable transfer support canceling logic https://github.com/elastic/kibana/issues/190485
    const contentInsightsClient = new _contentManagementContentInsightsPublic.ContentInsightsClient({
      http: _kibana_services.coreServices.http
    }, {
      domainId: 'dashboard'
    });
    contentInsightsClient.track(loadDashboardReturn.dashboardId, 'viewed');
  }
  return {
    input: initialDashboardInput,
    searchSessionId: initialSearchSessionId
  };
};
exports.initializeDashboard = initializeDashboard;