"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertPanelMapToSavedPanels = void 0;
exports.convertPanelStateToSavedDashboardPanel = convertPanelStateToSavedDashboardPanel;
exports.convertSavedDashboardPanelToPanelState = convertSavedDashboardPanelToPanelState;
exports.generateNewPanelIds = exports.convertSavedPanelsToPanelMap = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _dashboard_container_references = require("../dashboard_container/persistable_state/dashboard_container_references");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function convertSavedDashboardPanelToPanelState(savedDashboardPanel) {
  return {
    type: savedDashboardPanel.type,
    gridData: savedDashboardPanel.gridData,
    panelRefName: savedDashboardPanel.panelRefName,
    explicitInput: {
      id: savedDashboardPanel.panelIndex,
      ...(savedDashboardPanel.id !== undefined && {
        savedObjectId: savedDashboardPanel.id
      }),
      ...(savedDashboardPanel.title !== undefined && {
        title: savedDashboardPanel.title
      }),
      ...savedDashboardPanel.embeddableConfig
    },
    /**
     * Version information used to be stored in the panel until 8.11 when it was moved
     * to live inside the explicit Embeddable Input. If version information is given here, we'd like to keep it.
     * It will be removed on Dashboard save
     */
    version: savedDashboardPanel.version
  };
}
function convertPanelStateToSavedDashboardPanel(panelState, removeLegacyVersion) {
  const savedObjectId = panelState.explicitInput.savedObjectId;
  return {
    /**
     * Version information used to be stored in the panel until 8.11 when it was moved to live inside the
     * explicit Embeddable Input. If removeLegacyVersion is not passed, we'd like to keep this information for
     * the time being.
     */
    ...(!removeLegacyVersion ? {
      version: panelState.version
    } : {}),
    type: panelState.type,
    gridData: panelState.gridData,
    panelIndex: panelState.explicitInput.id,
    embeddableConfig: (0, _lodash.omit)(panelState.explicitInput, ['id', 'savedObjectId', 'title']),
    ...(panelState.explicitInput.title !== undefined && {
      title: panelState.explicitInput.title
    }),
    ...(savedObjectId !== undefined && {
      id: savedObjectId
    }),
    ...(panelState.panelRefName !== undefined && {
      panelRefName: panelState.panelRefName
    })
  };
}
const convertSavedPanelsToPanelMap = panels => {
  const panelsMap = {};
  panels === null || panels === void 0 ? void 0 : panels.forEach((panel, idx) => {
    var _panel$panelIndex;
    panelsMap[(_panel$panelIndex = panel.panelIndex) !== null && _panel$panelIndex !== void 0 ? _panel$panelIndex : String(idx)] = convertSavedDashboardPanelToPanelState(panel);
  });
  return panelsMap;
};
exports.convertSavedPanelsToPanelMap = convertSavedPanelsToPanelMap;
const convertPanelMapToSavedPanels = (panels, removeLegacyVersion) => {
  return Object.values(panels).map(panel => convertPanelStateToSavedDashboardPanel(panel, removeLegacyVersion));
};

/**
 * When saving a dashboard as a copy, we should generate new IDs for all panels so that they are
 * properly refreshed when navigating between Dashboards
 */
exports.convertPanelMapToSavedPanels = convertPanelMapToSavedPanels;
const generateNewPanelIds = (panels, references) => {
  const newPanelsMap = {};
  const newReferences = [];
  for (const [oldId, panel] of Object.entries(panels)) {
    const newId = (0, _uuid.v4)();
    newPanelsMap[newId] = {
      ...panel,
      gridData: {
        ...panel.gridData,
        i: newId
      },
      explicitInput: {
        ...panel.explicitInput,
        id: newId
      }
    };
    newReferences.push(...(0, _dashboard_container_references.prefixReferencesFromPanel)(newId, (0, _dashboard_container_references.getReferencesForPanelId)(oldId, references !== null && references !== void 0 ? references : [])));
  }
  return {
    panels: newPanelsMap,
    references: newReferences
  };
};
exports.generateNewPanelIds = generateNewPanelIds;