"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerStatusRoute = void 0;
var _rxjs = require("rxjs");
var _configSchema = require("@kbn/config-schema");
var _coreStatusCommon = require("@kbn/core-status-common");
var _legacy_status = require("../legacy_status");
var _status_response_schemas = require("./status_response_schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SNAPSHOT_POSTFIX = /-SNAPSHOT$/;

// The moment we remove support for the LegacyStatusInfo, we can use the StatusResponse straight away.

const SERVICE_UNAVAILABLE_NOT_REPORTED = {
  level: _coreStatusCommon.ServiceStatusLevels.unavailable,
  summary: 'Status not yet reported'
};
const registerStatusRoute = ({
  router,
  config,
  metrics,
  status,
  incrementUsageCounter
}) => {
  // Since the status.plugins$ observable is not subscribed to elsewhere, we need to subscribe it here to eagerly load
  // the plugins status when Kibana starts up so this endpoint responds quickly on first boot.
  const combinedStatus$ = new _rxjs.ReplaySubject(1);
  (0, _rxjs.combineLatest)([status.overall$.pipe((0, _rxjs.startWith)(SERVICE_UNAVAILABLE_NOT_REPORTED)), status.coreOverall$.pipe((0, _rxjs.startWith)(SERVICE_UNAVAILABLE_NOT_REPORTED)), status.core$.pipe((0, _rxjs.startWith)({
    elasticsearch: SERVICE_UNAVAILABLE_NOT_REPORTED,
    savedObjects: SERVICE_UNAVAILABLE_NOT_REPORTED
  })), status.plugins$.pipe((0, _rxjs.startWith)({}))]).subscribe(combinedStatus$);
  router.get({
    path: '/api/status',
    options: {
      authRequired: 'optional',
      // The `api` tag ensures that unauthenticated calls receive a 401 rather than a 302 redirect to login page.
      // The `security:acceptJWT` tag allows route to be accessed with JWT credentials. It points to
      // ROUTE_TAG_ACCEPT_JWT from '@kbn/security-plugin/server' that cannot be imported here directly.
      tags: ['api', 'security:acceptJWT', 'oas-tag:system'],
      access: 'public',
      // needs to be public to allow access from "system" users like k8s readiness probes.
      summary: `Get Kibana's current status`
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          v7format: _configSchema.schema.maybe(_configSchema.schema.boolean({
            meta: {
              description: 'Set to "true" to get the response in v7 format.'
            }
          })),
          v8format: _configSchema.schema.maybe(_configSchema.schema.boolean({
            meta: {
              description: 'Set to "true" to get the response in v8 format.'
            }
          }))
        }, {
          validate: ({
            v7format,
            v8format
          }) => {
            if (typeof v7format === 'boolean' && typeof v8format === 'boolean') {
              return `provide only one format option: v7format or v8format`;
            }
          },
          meta: {
            description: `Return status in a specific format. If both v7 and v8 are requested the request will be rejected.`
          }
        })
      },
      response: {
        200: {
          description: 'Overall status is OK and Kibana should be functioning normally.',
          body: _status_response_schemas.statusResponse
        },
        503: {
          description: `Kibana or some of it's essential services are unavailable. Kibana may be degraded or unavailable.`,
          body: _status_response_schemas.statusResponse
        }
      }
    }
  }, async (context, req, res) => {
    const authRequired = !config.allowAnonymous;
    const isAuthenticated = req.auth.isAuthenticated;
    const redactedStatus = authRequired && !isAuthenticated;
    const [overall, coreOverall, core, plugins] = await (0, _rxjs.firstValueFrom)(combinedStatus$);
    const responseBody = redactedStatus ? getRedactedStatusResponse({
      coreOverall
    }) : await getFullStatusResponse({
      incrementUsageCounter,
      config,
      query: req.query,
      metrics,
      statuses: {
        overall,
        core,
        plugins
      }
    });
    const statusCode = coreOverall.level >= _coreStatusCommon.ServiceStatusLevels.unavailable ? 503 : 200;
    return res.custom({
      body: responseBody,
      statusCode,
      bypassErrorFormat: true
    });
  });
};
exports.registerStatusRoute = registerStatusRoute;
const getFullStatusResponse = async ({
  config,
  incrementUsageCounter,
  metrics,
  statuses: {
    plugins,
    overall,
    core
  },
  query: {
    v7format = false,
    v8format = true
  }
}) => {
  const {
    version,
    buildSha,
    buildNum,
    buildDate,
    buildFlavor
  } = config.packageInfo;
  const versionWithoutSnapshot = version.replace(SNAPSHOT_POSTFIX, '');
  let statusInfo;
  if (!v7format && v8format) {
    statusInfo = {
      overall,
      core,
      plugins
    };
  } else {
    incrementUsageCounter({
      counterName: 'status_v7format'
    });
    statusInfo = (0, _legacy_status.calculateLegacyStatus)({
      overall,
      core,
      plugins,
      versionWithoutSnapshot
    });
  }
  const lastMetrics = await (0, _rxjs.firstValueFrom)(metrics.getOpsMetrics$());
  const body = {
    name: config.serverName,
    uuid: config.uuid,
    version: {
      number: versionWithoutSnapshot,
      build_hash: buildSha,
      build_number: buildNum,
      build_snapshot: SNAPSHOT_POSTFIX.test(version),
      build_flavor: buildFlavor,
      build_date: buildDate.toISOString()
    },
    status: statusInfo,
    metrics: {
      last_updated: lastMetrics.collected_at.toISOString(),
      collection_interval_in_millis: metrics.collectionInterval,
      os: lastMetrics.os,
      process: lastMetrics.process,
      processes: lastMetrics.processes,
      response_times: lastMetrics.response_times,
      concurrent_connections: lastMetrics.concurrent_connections,
      requests: {
        ...lastMetrics.requests,
        status_codes: lastMetrics.requests.statusCodes
      },
      elasticsearch_client: lastMetrics.elasticsearch_client
    }
  };
  return body;
};
const getRedactedStatusResponse = ({
  coreOverall
}) => {
  const body = {
    status: {
      overall: {
        level: coreOverall.level.toString()
      }
    }
  };
  return body;
};