"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerBulkUpdateRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerBulkUpdateRoute = (router, {
  config,
  coreUsageData,
  logger,
  access
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.put({
    path: '/_bulk_update',
    options: {
      summary: `Update saved objects`,
      tags: ['oas-tag:saved objects'],
      access,
      deprecated: true
    },
    validate: {
      body: _configSchema.schema.arrayOf(_configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.string(),
        attributes: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any()),
        version: _configSchema.schema.maybe(_configSchema.schema.string()),
        references: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
          name: _configSchema.schema.string(),
          type: _configSchema.schema.string(),
          id: _configSchema.schema.string()
        }))),
        namespace: _configSchema.schema.maybe(_configSchema.schema.string({
          minLength: 1
        }))
      }))
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'put',
      path: '/api/saved_objects/_bulk_update',
      request,
      logger
    });
    const types = [...new Set(request.body.map(({
      type
    }) => type))];
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsBulkUpdate({
      request,
      types
    }).catch(() => {});
    const {
      savedObjects
    } = await context.core;
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfAnyTypeNotVisibleByAPI)(types, savedObjects.typeRegistry);
    }
    const savedObject = await savedObjects.client.bulkUpdate(request.body);
    return response.ok({
      body: savedObject
    });
  }));
};
exports.registerBulkUpdateRoute = registerBulkUpdateRoute;