"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performCheckConflicts = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performCheckConflicts = async ({
  objects,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  extensions = {}
}) => {
  const {
    common: commonHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  if (objects.length === 0) {
    return {
      errors: []
    };
  }
  let bulkGetRequestIndexCounter = 0;
  const expectedBulkGetResults = objects.map(object => {
    const {
      type,
      id
    } = object;
    if (!allowedTypes.includes(type)) {
      return (0, _utils.left)({
        id,
        type,
        error: (0, _utils.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type))
      });
    }
    return (0, _utils.right)({
      type,
      id,
      esRequestIndex: bulkGetRequestIndexCounter++
    });
  });
  const validObjects = expectedBulkGetResults.filter(_utils.isRight);
  await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeCheckConflicts({
    namespace,
    objects: validObjects.map(element => ({
      type: element.value.type,
      id: element.value.id
    }))
  }));
  const bulkGetDocs = validObjects.map(({
    value: {
      type,
      id
    }
  }) => ({
    _id: serializer.generateRawId(namespace, type, id),
    _index: commonHelper.getIndexForType(type),
    _source: {
      includes: ['type', 'namespaces']
    }
  }));
  const bulkGetResponse = bulkGetDocs.length ? await client.mget({
    body: {
      docs: bulkGetDocs
    }
  }, {
    ignore: [404],
    meta: true
  }) : undefined;
  // throw if we can't verify a 404 response is from Elasticsearch
  if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode: bulkGetResponse.statusCode,
    headers: bulkGetResponse.headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
  }
  const errors = [];
  expectedBulkGetResults.forEach(expectedResult => {
    if ((0, _utils.isLeft)(expectedResult)) {
      errors.push(expectedResult.value);
      return;
    }
    const {
      type,
      id,
      esRequestIndex
    } = expectedResult.value;
    const doc = bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex];
    if ((0, _utils.isMgetDoc)(doc) && doc.found) {
      errors.push({
        id,
        type,
        error: {
          ...(0, _utils.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createConflictError(type, id)),
          ...(!(0, _utils.rawDocExistsInNamespace)(registry, doc, namespace) && {
            metadata: {
              isNotOverwritable: true
            }
          })
        }
      });
    }
  });
  return {
    errors
  };
};
exports.performCheckConflicts = performCheckConflicts;