"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bootstrapRendererFactory = void 0;
var _crypto = require("crypto");
var _coreUiSettingsCommon = require("@kbn/core-ui-settings-common");
var _get_plugin_bundle_paths = require("./get_plugin_bundle_paths");
var _get_js_dependency_paths = require("./get_js_dependency_paths");
var _get_theme_tag = require("./get_theme_tag");
var _render_template = require("./render_template");
var _render_utils = require("../render_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const bootstrapRendererFactory = ({
  packageInfo,
  baseHref,
  uiPlugins,
  auth,
  userSettingsService
}) => {
  const isAuthenticated = request => {
    const {
      status: authStatus
    } = auth.get(request);
    // status is 'unknown' when auth is disabled. we just need to not be `unauthenticated` here.
    return authStatus !== 'unauthenticated';
  };
  return async function bootstrapRenderer({
    uiSettingsClient,
    request,
    isAnonymousPage = false
  }) {
    let darkMode = false;
    const themeVersion = 'v8';
    try {
      const authenticated = isAuthenticated(request);
      if (authenticated) {
        const userSettingDarkMode = await (userSettingsService === null || userSettingsService === void 0 ? void 0 : userSettingsService.getUserSettingDarkMode(request));
        if (userSettingDarkMode !== undefined) {
          darkMode = userSettingDarkMode;
        } else {
          darkMode = (0, _coreUiSettingsCommon.parseDarkModeValue)(await uiSettingsClient.get('theme:darkMode'));
        }
      }
    } catch (e) {
      // just use the default values in case of connectivity issues with ES
    }

    // keeping legacy themeTag support - note that the browser is now overriding it during setup.
    if (darkMode === 'system') {
      darkMode = false;
    }
    const themeTag = (0, _get_theme_tag.getThemeTag)({
      themeVersion,
      darkMode
    });
    const bundlesHref = (0, _render_utils.getBundlesHref)(baseHref);
    const bundlePaths = (0, _get_plugin_bundle_paths.getPluginsBundlePaths)({
      uiPlugins,
      bundlesHref,
      isAnonymousPage
    });
    const jsDependencyPaths = (0, _get_js_dependency_paths.getJsDependencyPaths)(bundlesHref, bundlePaths);

    // These paths should align with the bundle routes configured in
    // src/optimize/bundles_route/bundles_route.ts
    const publicPathMap = JSON.stringify({
      core: `${bundlesHref}/core/`,
      'kbn-ui-shared-deps-src': `${bundlesHref}/kbn-ui-shared-deps-src/`,
      'kbn-ui-shared-deps-npm': `${bundlesHref}/kbn-ui-shared-deps-npm/`,
      'kbn-monaco': `${bundlesHref}/kbn-monaco/`,
      ...Object.fromEntries([...bundlePaths.entries()].map(([pluginId, plugin]) => [pluginId, plugin.publicPath]))
    });
    const body = (0, _render_template.renderTemplate)({
      themeTag,
      jsDependencyPaths,
      publicPathMap
    });
    const hash = (0, _crypto.createHash)('sha1');
    hash.update(body);
    const etag = hash.digest('hex');
    return {
      body,
      etag
    };
  };
};
exports.bootstrapRendererFactory = bootstrapRendererFactory;