"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validRouteSecurity = void 0;
var _configSchema = require("@kbn/config-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const privilegeSetSchema = _configSchema.schema.object({
  anyRequired: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 2
  })),
  allRequired: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
    minSize: 1
  }))
}, {
  validate: value => {
    if (!value.anyRequired && !value.allRequired) {
      return 'either anyRequired or allRequired must be specified';
    }
  }
});
const requiredPrivilegesSchema = _configSchema.schema.arrayOf(_configSchema.schema.oneOf([privilegeSetSchema, _configSchema.schema.string()]), {
  validate: value => {
    const anyRequired = [];
    const allRequired = [];
    if (!Array.isArray(value)) {
      return undefined;
    }
    value.forEach(privilege => {
      if (typeof privilege === 'string') {
        allRequired.push(privilege);
      } else {
        if (privilege.anyRequired) {
          anyRequired.push(...privilege.anyRequired);
        }
        if (privilege.allRequired) {
          allRequired.push(...privilege.allRequired);
        }
      }
    });
    if (anyRequired.length && allRequired.length) {
      for (const privilege of anyRequired) {
        if (allRequired.includes(privilege)) {
          return `anyRequired and allRequired cannot have the same values: [${privilege}]`;
        }
      }
    }
    if (anyRequired.length) {
      const uniquePrivileges = new Set([...anyRequired]);
      if (anyRequired.length !== uniquePrivileges.size) {
        return 'anyRequired privileges must contain unique values';
      }
    }
  },
  minSize: 1
});
const authzSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.maybe(_configSchema.schema.literal(false)),
  requiredPrivileges: _configSchema.schema.conditional(_configSchema.schema.siblingRef('enabled'), _configSchema.schema.never(), requiredPrivilegesSchema, _configSchema.schema.never()),
  reason: _configSchema.schema.conditional(_configSchema.schema.siblingRef('enabled'), _configSchema.schema.never(), _configSchema.schema.never(), _configSchema.schema.string())
});
const authcSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.oneOf([_configSchema.schema.literal(true), _configSchema.schema.literal('optional'), _configSchema.schema.literal(false)]),
  reason: _configSchema.schema.conditional(_configSchema.schema.siblingRef('enabled'), _configSchema.schema.literal(false), _configSchema.schema.string(), _configSchema.schema.never())
});
const routeSecuritySchema = _configSchema.schema.object({
  authz: authzSchema,
  authc: _configSchema.schema.maybe(authcSchema)
});
const validRouteSecurity = (routeSecurity, options) => {
  if (!routeSecurity) {
    return routeSecurity;
  }
  if ((routeSecurity === null || routeSecurity === void 0 ? void 0 : routeSecurity.authc) !== undefined && (options === null || options === void 0 ? void 0 : options.authRequired) !== undefined) {
    throw new Error('Cannot specify both security.authc and options.authRequired');
  }
  return routeSecuritySchema.validate(routeSecurity);
};
exports.validRouteSecurity = validRouteSecurity;