"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Router = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _configSchema = require("@kbn/config-schema");
var _esErrors = require("@kbn/es-errors");
var _zod = require("@kbn/zod");
var _coreHttpServer = require("@kbn/core-http-server");
var _validator = require("./validator");
var _versioned_router = require("./versioned_router");
var _request = require("./request");
var _response = require("./response");
var _response_adapter = require("./response_adapter");
var _error_wrapper = require("./error_wrapper");
var _util = require("./util");
var _strip_illegal_http2_headers = require("./strip_illegal_http2_headers");
var _security_route_config_validator = require("./security_route_config_validator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getRouteFullPath(routerPath, routePath) {
  // If router's path ends with slash and route's path starts with slash,
  // we should omit one of them to have a valid concatenated path.
  const routePathStartIndex = routerPath.endsWith('/') && routePath.startsWith('/') ? 1 : 0;
  return `${routerPath}${routePath.slice(routePathStartIndex)}`;
}

/**
 * Create the validation schemas for a route
 *
 * @returns Route schemas if `validate` is specified on the route, otherwise
 * undefined.
 */
function routeSchemasFromRouteConfig(route, routeMethod) {
  // The type doesn't allow `validate` to be undefined, but it can still
  // happen when it's used from JavaScript.
  if (route.validate === undefined) {
    throw new Error(`The [${routeMethod}] at [${route.path}] does not have a 'validate' specified. Use 'false' as the value if you want to bypass validation.`);
  }
  if (route.validate !== false) {
    const validation = (0, _coreHttpServer.getRequestValidation)(route.validate);
    Object.entries(validation).forEach(([key, schema]) => {
      if (!((0, _configSchema.isConfigSchema)(schema) || (0, _zod.isZod)(schema) || typeof schema === 'function')) {
        throw new Error(`Expected a valid validation logic declared with '@kbn/config-schema' package, '@kbn/zod' package or a RouteValidationFunction at key: [${key}].`);
      }
    });
    return _validator.RouteValidator.from(validation);
  }
}

/**
 * Create a valid options object with "sensible" defaults + adding some validation to the options fields
 *
 * @param method HTTP verb for these options
 * @param routeConfig The route config definition
 */
function validOptions(method, routeConfig) {
  const shouldNotHavePayload = ['head', 'get'].includes(method);
  const {
    options = {},
    validate
  } = routeConfig;
  const shouldValidateBody = validate && !!(0, _coreHttpServer.getRequestValidation)(validate).body || !!options.body;
  const {
    output
  } = options.body || {};
  if (typeof output === 'string' && !_coreHttpServer.validBodyOutput.includes(output)) {
    throw new Error(`[options.body.output: '${output}'] in route ${method.toUpperCase()} ${routeConfig.path} is not valid. Only '${_coreHttpServer.validBodyOutput.join("' or '")}' are valid.`);
  }
  const body = shouldNotHavePayload ? undefined : {
    // If it's not a GET (requires payload) but no body validation is required (or no body options are specified),
    // We assume the route does not care about the body => use the memory-cheapest approach (stream and no parsing)
    output: !shouldValidateBody ? 'stream' : undefined,
    parse: !shouldValidateBody ? false : undefined,
    // User's settings should overwrite any of the "desired" values
    ...options.body
  };
  return {
    ...options,
    body
  };
}

/** @internal */

/** @internal */

/** @internal */

/** @internal */

/** @internal */

/** @internal */

/**
 * @internal
 */
class Router {
  constructor(routerPath, log, enhanceWithContext, options) {
    (0, _defineProperty2.default)(this, "routes", []);
    (0, _defineProperty2.default)(this, "pluginId", void 0);
    (0, _defineProperty2.default)(this, "get", void 0);
    (0, _defineProperty2.default)(this, "post", void 0);
    (0, _defineProperty2.default)(this, "delete", void 0);
    (0, _defineProperty2.default)(this, "put", void 0);
    (0, _defineProperty2.default)(this, "patch", void 0);
    (0, _defineProperty2.default)(this, "handleLegacyErrors", _error_wrapper.wrapErrors);
    (0, _defineProperty2.default)(this, "versionedRouter", undefined);
    this.routerPath = routerPath;
    this.log = log;
    this.enhanceWithContext = enhanceWithContext;
    this.options = options;
    this.pluginId = options.pluginId;
    const buildMethod = method => (route, handler, {
      isVersioned
    } = {
      isVersioned: false
    }) => {
      var _route$options, _route$options2;
      route = (0, _util.prepareRouteConfigValidation)(route);
      const routeSchemas = routeSchemasFromRouteConfig(route, method);
      const isPublicUnversionedApi = !isVersioned && ((_route$options = route.options) === null || _route$options === void 0 ? void 0 : _route$options.access) === 'public' &&
      // We do not consider HTTP resource routes as APIs
      ((_route$options2 = route.options) === null || _route$options2 === void 0 ? void 0 : _route$options2.httpResource) !== true;
      this.routes.push({
        handler: async (req, responseToolkit) => await this.handle({
          routeSchemas,
          request: req,
          responseToolkit,
          isPublicUnversionedApi,
          handler: this.enhanceWithContext(handler)
        }),
        method,
        path: getRouteFullPath(this.routerPath, route.path),
        options: validOptions(method, route),
        // For the versioned route security is validated in the versioned router
        security: isVersioned ? route.security : (0, _security_route_config_validator.validRouteSecurity)(route.security, route.options),
        validationSchemas: route.validate,
        isVersioned
      });
    };
    this.get = buildMethod('get');
    this.post = buildMethod('post');
    this.delete = buildMethod('delete');
    this.put = buildMethod('put');
    this.patch = buildMethod('patch');
  }
  getRoutes({
    excludeVersionedRoutes
  } = {}) {
    if (excludeVersionedRoutes) {
      return this.routes.filter(route => !route.isVersioned);
    }
    return [...this.routes];
  }
  logError(msg, statusCode, {
    error,
    request
  }) {
    var _request$route, _request$route2;
    this.log.error(msg, {
      http: {
        response: {
          status_code: statusCode
        },
        request: {
          method: (_request$route = request.route) === null || _request$route === void 0 ? void 0 : _request$route.method,
          path: (_request$route2 = request.route) === null || _request$route2 === void 0 ? void 0 : _request$route2.path
        }
      },
      error: {
        message: error.message
      }
    });
  }
  async handle({
    routeSchemas,
    request,
    responseToolkit,
    isPublicUnversionedApi,
    handler
  }) {
    let kibanaRequest;
    const hapiResponseAdapter = new _response_adapter.HapiResponseAdapter(responseToolkit);
    try {
      kibanaRequest = _request.CoreKibanaRequest.from(request, routeSchemas);
    } catch (error) {
      this.logError('400 Bad Request', 400, {
        request,
        error
      });
      const response = hapiResponseAdapter.toBadRequest(error.message);
      if (isPublicUnversionedApi) {
        response.output.headers = {
          ...response.output.headers,
          ...(0, _util.getVersionHeader)(_versioned_router.ALLOWED_PUBLIC_VERSION)
        };
      }
      return response;
    }
    try {
      const kibanaResponse = await handler(kibanaRequest, _response.kibanaResponseFactory);
      if (isPublicUnversionedApi) {
        (0, _util.injectVersionHeader)(_versioned_router.ALLOWED_PUBLIC_VERSION, kibanaResponse);
      }
      if (kibanaRequest.protocol === 'http2' && kibanaResponse.options.headers) {
        var _this$options$isDev;
        kibanaResponse.options.headers = (0, _strip_illegal_http2_headers.stripIllegalHttp2Headers)({
          headers: kibanaResponse.options.headers,
          isDev: (_this$options$isDev = this.options.isDev) !== null && _this$options$isDev !== void 0 ? _this$options$isDev : false,
          logger: this.log,
          requestContext: `${request.route.method} ${request.route.path}`
        });
      }
      return hapiResponseAdapter.handle(kibanaResponse);
    } catch (error) {
      // capture error
      _elasticApmNode.default.captureError(error);

      // forward 401 errors from ES client
      if ((0, _esErrors.isUnauthorizedError)(error)) {
        this.logError('401 Unauthorized', 401, {
          request,
          error
        });
        return hapiResponseAdapter.handle(_response.kibanaResponseFactory.unauthorized(convertEsUnauthorized(error)));
      }

      // return a generic 500 to avoid error info / stack trace surfacing
      this.logError('500 Server Error', 500, {
        request,
        error
      });
      return hapiResponseAdapter.toInternalError();
    }
  }
  get versioned() {
    if (this.versionedRouter === undefined) {
      this.versionedRouter = _versioned_router.CoreVersionedRouter.from({
        router: this,
        isDev: this.options.isDev,
        ...this.options.versionedRouterOptions
      });
    }
    return this.versionedRouter;
  }
}
exports.Router = Router;
const convertEsUnauthorized = e => {
  const getAuthenticateHeaderValue = () => {
    const header = Object.entries(e.headers || {}).find(([key]) => key.toLowerCase() === 'www-authenticate');
    return header ? header[1] : 'Basic realm="Authorization Required"';
  };
  return {
    body: e.message,
    headers: {
      'www-authenticate': getAuthenticateHeaderValue()
    }
  };
};