"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FeatureFlagsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _serverSdk = require("@openfeature/server-sdk");
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _rxjs = require("rxjs");
var _feature_flags_config = require("./feature_flags_config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Core-internal contract for the setup lifecycle step.
 * @private
 */

/**
 * The server-side Feature Flags Service
 * @private
 */
class FeatureFlagsService {
  /**
   * The core service's constructor
   * @param core {@link CoreContext}
   */
  constructor(core) {
    (0, _defineProperty2.default)(this, "featureFlagsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "overrides", {});
    (0, _defineProperty2.default)(this, "context", {
      kind: 'multi'
    });
    this.core = core;
    this.logger = core.logger.get('feature-flags-service');
    this.featureFlagsClient = _serverSdk.OpenFeature.getClient();
    _serverSdk.OpenFeature.setLogger(this.logger.get('open-feature'));
  }

  /**
   * Setup lifecycle method
   */
  setup() {
    // Register "overrides" to be changed via the dynamic config endpoint (enabled in test environments only)
    this.core.configService.addDynamicConfigPaths(_feature_flags_config.featureFlagsConfig.path, ['overrides']);
    this.core.configService.atPath(_feature_flags_config.featureFlagsConfig.path).subscribe(({
      overrides = {}
    }) => {
      this.overrides = overrides;
    });
    return {
      getOverrides: () => this.overrides,
      setProvider: provider => {
        if (_serverSdk.OpenFeature.providerMetadata !== _serverSdk.NOOP_PROVIDER.metadata) {
          throw new Error('A provider has already been set. This API cannot be called twice.');
        }
        _serverSdk.OpenFeature.setProvider(provider);
      },
      appendContext: contextToAppend => this.appendContext(contextToAppend)
    };
  }

  /**
   * Start lifecycle method
   */
  start() {
    const featureFlagsChanged$ = new _rxjs.Subject();
    this.featureFlagsClient.addHandler(_serverSdk.ServerProviderEvents.ConfigurationChanged, event => {
      if (event !== null && event !== void 0 && event.flagsChanged) {
        featureFlagsChanged$.next(event.flagsChanged);
      }
    });
    const observeFeatureFlag$ = flagName => featureFlagsChanged$.pipe((0, _rxjs.filter)(flagNames => flagNames.includes(flagName)), (0, _rxjs.startWith)([flagName]) // only to emit on the first call
    );
    return {
      appendContext: contextToAppend => this.appendContext(contextToAppend),
      getBooleanValue: async (flagName, fallbackValue) => this.evaluateFlag(this.featureFlagsClient.getBooleanValue, flagName, fallbackValue),
      getStringValue: async (flagName, fallbackValue) => await this.evaluateFlag(this.featureFlagsClient.getStringValue, flagName, fallbackValue),
      getNumberValue: async (flagName, fallbackValue) => await this.evaluateFlag(this.featureFlagsClient.getNumberValue, flagName, fallbackValue),
      getBooleanValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.switchMap)(() => this.evaluateFlag(this.featureFlagsClient.getBooleanValue, flagName, fallbackValue)));
      },
      getStringValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.switchMap)(() => this.evaluateFlag(this.featureFlagsClient.getStringValue, flagName, fallbackValue)));
      },
      getNumberValue$: (flagName, fallbackValue) => {
        return observeFeatureFlag$(flagName).pipe((0, _rxjs.switchMap)(() => this.evaluateFlag(this.featureFlagsClient.getNumberValue, flagName, fallbackValue)));
      }
    };
  }

  /**
   * Stop lifecycle method
   */
  async stop() {
    await _serverSdk.OpenFeature.close();
  }

  /**
   * Wrapper to evaluate flags with the common config overrides interceptions + APM and counters reporting
   * @param evaluationFn The actual evaluation API
   * @param flagName The name of the flag to evaluate
   * @param fallbackValue The fallback value
   * @private
   */
  async evaluateFlag(evaluationFn, flagName, fallbackValue) {
    const value = typeof this.overrides[flagName] !== 'undefined' ? this.overrides[flagName] :
    // We have to bind the evaluation or the client will lose its internal context
    await evaluationFn.bind(this.featureFlagsClient)(flagName, fallbackValue);
    _elasticApmNode.default.addLabels({
      [`flag_${flagName}`]: value
    });
    // TODO: increment usage counter
    return value;
  }

  /**
   * Formats the provided context to fulfill the expected multi-context structure.
   * @param contextToAppend The {@link EvaluationContext} to append.
   * @private
   */
  appendContext(contextToAppend) {
    // If no kind provided, default to the project|deployment level.
    const {
      kind = 'kibana',
      ...rest
    } = contextToAppend;
    // Format the context to fulfill the expected multi-context structure
    const formattedContextToAppend = kind === 'multi' ? contextToAppend : {
      kind: 'multi',
      [kind]: rest
    };

    // Merge the formatted context to append to the global context, and set it in the OpenFeature client.
    this.context = (0, _deepmerge.default)(this.context, formattedContextToAppend);
    _serverSdk.OpenFeature.setContext(this.context);
  }
}
exports.FeatureFlagsService = FeatureFlagsService;