"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initTimeRangePercentage = initTimeRangePercentage;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _time_utils = require("./time_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initTimeRangePercentage(state, onReset) {
  const timesliceStartAsPercentageOfTimeRange$ = new _rxjs.BehaviorSubject(state.timesliceStartAsPercentageOfTimeRange);
  const timesliceEndAsPercentageOfTimeRange$ = new _rxjs.BehaviorSubject(state.timesliceEndAsPercentageOfTimeRange);

  // debounce to avoid calling 'resetTimeslice' on each comparator reset
  const debouncedOnReset = (0, _lodash.debounce)(() => {
    onReset(timesliceStartAsPercentageOfTimeRange$.value, timesliceEndAsPercentageOfTimeRange$.value);
  }, 0);
  return {
    setTimeRangePercentage(timeslice, timeRangeMeta) {
      let timesliceStartAsPercentageOfTimeRange;
      let timesliceEndAsPercentageOfTimeRange;
      if (timeslice) {
        timesliceStartAsPercentageOfTimeRange = (timeslice[_time_utils.FROM_INDEX] - timeRangeMeta.timeRangeBounds[_time_utils.FROM_INDEX]) / timeRangeMeta.timeRange;
        timesliceEndAsPercentageOfTimeRange = (timeslice[_time_utils.TO_INDEX] - timeRangeMeta.timeRangeBounds[_time_utils.FROM_INDEX]) / timeRangeMeta.timeRange;
      }
      timesliceStartAsPercentageOfTimeRange$.next(timesliceStartAsPercentageOfTimeRange);
      timesliceEndAsPercentageOfTimeRange$.next(timesliceEndAsPercentageOfTimeRange);
    },
    serializeState: () => {
      return {
        timesliceStartAsPercentageOfTimeRange: timesliceStartAsPercentageOfTimeRange$.value,
        timesliceEndAsPercentageOfTimeRange: timesliceEndAsPercentageOfTimeRange$.value
      };
    },
    comparators: {
      timesliceStartAsPercentageOfTimeRange: [timesliceStartAsPercentageOfTimeRange$, value => {
        timesliceStartAsPercentageOfTimeRange$.next(value);
        debouncedOnReset();
      }],
      timesliceEndAsPercentageOfTimeRange: [timesliceEndAsPercentageOfTimeRange$, value => {
        timesliceEndAsPercentageOfTimeRange$.next(value);
        debouncedOnReset();
      }]
    }
  };
}