"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeDataControl = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _common = require("@kbn/data-views-plugin/common");
var _i18n = require("@kbn/i18n");
var _kibana_services = require("../../services/kibana_services");
var _initialize_default_control_api = require("../initialize_default_control_api");
var _open_data_control_editor = require("./open_data_control_editor");
var _reference_name_utils = require("./reference_name_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const initializeDataControl = (controlId, controlType, referenceNameSuffix, state, editorStateManager, controlGroupApi) => {
  const defaultControl = (0, _initialize_default_control_api.initializeDefaultControlApi)(state);
  const panelTitle = new _rxjs.BehaviorSubject(state.title);
  const defaultPanelTitle = new _rxjs.BehaviorSubject(undefined);
  const dataViewId = new _rxjs.BehaviorSubject(state.dataViewId);
  const fieldName = new _rxjs.BehaviorSubject(state.fieldName);
  const dataViews = new _rxjs.BehaviorSubject(undefined);
  const filters$ = new _rxjs.BehaviorSubject(undefined);
  const filtersReady$ = new _rxjs.BehaviorSubject(false);
  const field$ = new _rxjs.BehaviorSubject(undefined);
  const fieldFormatter = new _rxjs.BehaviorSubject(toFormat => String(toFormat));
  const stateManager = {
    ...defaultControl.stateManager,
    dataViewId,
    fieldName,
    title: panelTitle
  };
  const dataViewIdSubscription = dataViewId.pipe((0, _rxjs.tap)(() => {
    filtersReady$.next(false);
    if (defaultControl.api.blockingError.value) {
      defaultControl.api.setBlockingError(undefined);
    }
  }), (0, _rxjs.switchMap)(async currentDataViewId => {
    let dataView;
    try {
      dataView = await _kibana_services.dataViewsService.get(currentDataViewId);
      return {
        dataView
      };
    } catch (error) {
      return {
        error
      };
    }
  })).subscribe(({
    dataView,
    error
  }) => {
    if (error) {
      defaultControl.api.setBlockingError(error);
    }
    dataViews.next(dataView ? [dataView] : undefined);
  });
  const fieldNameSubscription = (0, _rxjs.combineLatest)([dataViews, fieldName]).pipe((0, _rxjs.tap)(() => {
    filtersReady$.next(false);
  })).subscribe(([nextDataViews, nextFieldName]) => {
    const dataView = nextDataViews ? nextDataViews.find(({
      id
    }) => dataViewId.value === id) : undefined;
    if (!dataView) {
      return;
    }
    const field = dataView.getFieldByName(nextFieldName);
    if (!field) {
      defaultControl.api.setBlockingError(new Error(_i18n.i18n.translate('controls.dataControl.fieldNotFound', {
        defaultMessage: 'Could not locate field: {fieldName}',
        values: {
          fieldName: nextFieldName
        }
      })));
    } else if (defaultControl.api.blockingError.value) {
      defaultControl.api.setBlockingError(undefined);
    }
    field$.next(field);
    defaultPanelTitle.next(field ? field.displayName || field.name : nextFieldName);
    const spec = field === null || field === void 0 ? void 0 : field.toSpec();
    if (spec) {
      fieldFormatter.next(dataView.getFormatterForField(spec).getConverterFor('text'));
    }
  });
  const onEdit = async () => {
    // get the initial state from the state manager
    const mergedStateManager = {
      ...stateManager,
      ...editorStateManager
    };
    const initialState = Object.keys(mergedStateManager).reduce((prev, key) => {
      var _mergedStateManager$k;
      return {
        ...prev,
        [key]: (_mergedStateManager$k = mergedStateManager[key]) === null || _mergedStateManager$k === void 0 ? void 0 : _mergedStateManager$k.getValue()
      };
    }, {});

    // open the editor to get the new state
    (0, _open_data_control_editor.openDataControlEditor)({
      onSave: ({
        type: newType,
        state: newState
      }) => {
        if (newType === controlType) {
          // apply the changes from the new state via the state manager
          Object.keys(initialState).forEach(key => {
            if (!(0, _lodash.isEqual)(mergedStateManager[key].getValue(), newState[key])) {
              mergedStateManager[key].next(newState[key]);
            }
          });
        } else {
          // replace the control with a new one of the updated type
          controlGroupApi.replacePanel(controlId, {
            panelType: newType,
            initialState: newState
          });
        }
      },
      initialState: {
        ...initialState
      },
      controlType,
      controlId,
      initialDefaultPanelTitle: defaultPanelTitle.getValue(),
      controlGroupApi
    });
  };
  const filtersReadySubscription = filters$.pipe((0, _rxjs.skip)(1), (0, _rxjs.debounceTime)(0)).subscribe(() => {
    // Set filtersReady$.next(true); in filters$ subscription instead of setOutputFilter
    // to avoid signaling filters ready until after filters have been emitted
    // to avoid timing issues
    filtersReady$.next(true);
  });
  const api = {
    ...defaultControl.api,
    panelTitle,
    defaultPanelTitle,
    dataViews,
    field$,
    fieldFormatter,
    onEdit,
    filters$,
    isEditingEnabled: () => true,
    untilFiltersReady: async () => {
      return new Promise(resolve => {
        (0, _rxjs.combineLatest)([defaultControl.api.blockingError, filtersReady$]).pipe((0, _rxjs.first)(([blockingError, filtersReady]) => filtersReady || blockingError !== undefined)).subscribe(() => {
          resolve();
        });
      });
    }
  };
  return {
    api,
    cleanup: () => {
      dataViewIdSubscription.unsubscribe();
      fieldNameSubscription.unsubscribe();
      filtersReadySubscription.unsubscribe();
    },
    comparators: {
      ...defaultControl.comparators,
      title: [panelTitle, value => panelTitle.next(value)],
      dataViewId: [dataViewId, value => dataViewId.next(value)],
      fieldName: [fieldName, value => fieldName.next(value)]
    },
    setters: {
      onSelectionChange: () => {
        filtersReady$.next(false);
      },
      setOutputFilter: newFilter => {
        filters$.next(newFilter ? [newFilter] : undefined);
      }
    },
    stateManager,
    serialize: () => {
      return {
        rawState: {
          ...defaultControl.serialize().rawState,
          dataViewId: dataViewId.getValue(),
          fieldName: fieldName.getValue(),
          title: panelTitle.getValue()
        },
        references: [{
          name: (0, _reference_name_utils.getReferenceName)(controlId, referenceNameSuffix),
          type: _common.DATA_VIEW_SAVED_OBJECT_TYPE,
          id: dataViewId.getValue()
        }]
      };
    }
  };
};
exports.initializeDataControl = initializeDataControl;