"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFavoritesRoutes = registerFavoritesRoutes;
var _server = require("@kbn/core/server");
var _configSchema = require("@kbn/config-schema");
var _favorites_service = require("./favorites_service");
var _favorites_saved_object = require("./favorites_saved_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// only dashboard is supported for now
// TODO: make configurable or allow any string
const typeSchema = _configSchema.schema.oneOf([_configSchema.schema.literal('dashboard')]);

/**
 * @public
 * Response for get favorites API
 */

function registerFavoritesRoutes({
  core,
  logger
}) {
  const router = core.http.createRouter();
  const getSavedObjectClient = coreRequestHandlerContext => {
    // We need to exclude security extension to access the `favorite` type which not every user has access to
    // and give access only to the current user's favorites through this API
    return coreRequestHandlerContext.savedObjects.getClient({
      includedHiddenTypes: [_favorites_saved_object.favoritesSavedObjectType.name],
      excludedExtensions: [_server.SECURITY_EXTENSION_ID]
    });
  };
  router.post({
    path: '/internal/content_management/favorites/{type}/{id}/favorite',
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        type: typeSchema
      })
    }
    // we don't protect the route with any access tags as
    // we only give access to the current user's favorites ids
  }, async (requestHandlerContext, request, response) => {
    var _coreRequestHandlerCo;
    const coreRequestHandlerContext = await requestHandlerContext.core;
    const userId = (_coreRequestHandlerCo = coreRequestHandlerContext.security.authc.getCurrentUser()) === null || _coreRequestHandlerCo === void 0 ? void 0 : _coreRequestHandlerCo.profile_uid;
    if (!userId) {
      return response.forbidden();
    }
    const type = request.params.type;
    const favorites = new _favorites_service.FavoritesService(type, userId, {
      savedObjectClient: getSavedObjectClient(coreRequestHandlerContext),
      logger
    });
    const favoriteIds = await favorites.addFavorite({
      id: request.params.id
    });
    return response.ok({
      body: favoriteIds
    });
  });
  router.post({
    path: '/internal/content_management/favorites/{type}/{id}/unfavorite',
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        type: typeSchema
      })
    }
    // we don't protect the route with any access tags as
    // we only give access to the current user's favorites ids
  }, async (requestHandlerContext, request, response) => {
    var _coreRequestHandlerCo2;
    const coreRequestHandlerContext = await requestHandlerContext.core;
    const userId = (_coreRequestHandlerCo2 = coreRequestHandlerContext.security.authc.getCurrentUser()) === null || _coreRequestHandlerCo2 === void 0 ? void 0 : _coreRequestHandlerCo2.profile_uid;
    if (!userId) {
      return response.forbidden();
    }
    const type = request.params.type;
    const favorites = new _favorites_service.FavoritesService(type, userId, {
      savedObjectClient: getSavedObjectClient(coreRequestHandlerContext),
      logger
    });
    const favoriteIds = await favorites.removeFavorite({
      id: request.params.id
    });
    return response.ok({
      body: favoriteIds
    });
  });
  router.get({
    path: '/internal/content_management/favorites/{type}',
    validate: {
      params: _configSchema.schema.object({
        type: typeSchema
      })
    }
    // we don't protect the route with any access tags as
    // we only give access to the current user's favorites ids
  }, async (requestHandlerContext, request, response) => {
    var _coreRequestHandlerCo3;
    const coreRequestHandlerContext = await requestHandlerContext.core;
    const userId = (_coreRequestHandlerCo3 = coreRequestHandlerContext.security.authc.getCurrentUser()) === null || _coreRequestHandlerCo3 === void 0 ? void 0 : _coreRequestHandlerCo3.profile_uid;
    if (!userId) {
      return response.forbidden();
    }
    const type = request.params.type;
    const favorites = new _favorites_service.FavoritesService(type, userId, {
      savedObjectClient: getSavedObjectClient(coreRequestHandlerContext),
      logger
    });
    const getFavoritesResponse = await favorites.getFavorites();
    return response.ok({
      body: getFavoritesResponse
    });
  });
}