"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VariablesEditor = void 0;
var _react = _interopRequireWildcard(require("react"));
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _eui = require("@elastic/eui");
var _variables_editor_form = require("./variables_editor_form");
var utils = _interopRequireWildcard(require("./utils"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const VariablesEditor = props => {
  const isMounted = (0, _react.useRef)(false);
  const [isAddingVariable, setIsAddingVariable] = (0, _react.useState)(false);
  const [deleteModalForVariable, setDeleteModalForVariable] = (0, _react.useState)(null);
  const [variables, setVariables] = (0, _react.useState)(props.variables);
  const deleteModalTitleId = (0, _eui.useGeneratedHtmlId)();

  // Use a ref to persist the BehaviorSubject across renders
  const itemIdToExpandedRowMap$ = (0, _react.useRef)(new _rxjs.BehaviorSubject({}));
  // Subscribe to the BehaviorSubject and update local state on change
  const [itemIdToExpandedRowMap, setItemIdToExpandedRowMap] = (0, _react.useState)({});
  // Clear the expanded row map and dispose all the expanded rows
  const collapseExpandedRows = () => itemIdToExpandedRowMap$.current.next({});

  // Subscribe to the BehaviorSubject on mount
  (0, _react.useEffect)(() => {
    const subscription = itemIdToExpandedRowMap$.current.subscribe(setItemIdToExpandedRowMap);
    return () => subscription.unsubscribe();
  }, []);

  // Always save variables when they change
  (0, _react.useEffect)(() => {
    if (isMounted.current) {
      props.onSaveVariables(variables);
    } else {
      isMounted.current = true;
    }
  }, [variables, props]);
  const toggleDetails = variableId => {
    const currentMap = itemIdToExpandedRowMap$.current.getValue();
    let itemIdToExpandedRowMapValues = {
      ...currentMap
    };
    if (itemIdToExpandedRowMapValues[variableId]) {
      delete itemIdToExpandedRowMapValues[variableId];
    } else {
      // Always close the add variable form when editing a variable
      setIsAddingVariable(false);
      // We only allow one expanded row at a time
      itemIdToExpandedRowMapValues = {};
      itemIdToExpandedRowMapValues[variableId] = /*#__PURE__*/_react.default.createElement(_variables_editor_form.VariableEditorForm, {
        title: _i18n.i18n.translate('console.variablesPage.editVariableForm.title', {
          defaultMessage: 'Edit variable'
        }),
        onSubmit: data => {
          const updatedVariables = utils.editVariable(data, variables);
          setVariables(updatedVariables);
          collapseExpandedRows();
        },
        onCancel: () => {
          collapseExpandedRows();
        },
        defaultValue: variables.find(v => v.id === variableId)
      });
    }

    // Update the BehaviorSubject with the new state
    itemIdToExpandedRowMap$.current.next(itemIdToExpandedRowMapValues);
  };
  const deleteVariable = (0, _react.useCallback)(id => {
    const updatedVariables = utils.deleteVariable(variables, id);
    setVariables(updatedVariables);
    setDeleteModalForVariable(null);
  }, [variables, setDeleteModalForVariable]);
  const onAddVariable = data => {
    setVariables(v => [...v, data]);
    setIsAddingVariable(false);
  };
  const columns = [{
    field: 'name',
    name: _i18n.i18n.translate('console.variablesPage.variablesTable.columns.variableHeader', {
      defaultMessage: 'Variable name'
    }),
    'data-test-subj': 'variableNameCell',
    render: name => {
      return /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, `\$\{${name}\}`);
    }
  }, {
    field: 'value',
    name: _i18n.i18n.translate('console.variablesPage.variablesTable.columns.valueHeader', {
      defaultMessage: 'Value'
    }),
    'data-test-subj': 'variableValueCell',
    render: value => /*#__PURE__*/_react.default.createElement(_eui.EuiCode, null, value)
  }, {
    field: 'id',
    name: '',
    width: '40px',
    isExpander: true,
    render: (id, variable) => {
      const itemIdToExpandedRowMapValues = {
        ...itemIdToExpandedRowMap
      };
      return /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
        iconType: itemIdToExpandedRowMapValues[id] ? 'arrowUp' : 'pencil',
        "aria-label": _i18n.i18n.translate('console.variablesPage.variablesTable.columns.editButton', {
          defaultMessage: 'Edit {variable}',
          values: {
            variable: variable.name
          }
        }),
        color: "primary",
        onClick: () => toggleDetails(id),
        "data-test-subj": "variableEditButton"
      });
    }
  }, {
    field: 'id',
    name: '',
    width: '40px',
    render: (id, variable) => /*#__PURE__*/_react.default.createElement(_eui.EuiButtonIcon, {
      iconType: "trash",
      "aria-label": _i18n.i18n.translate('console.variablesPage.variablesTable.columns.deleteButton', {
        defaultMessage: 'Delete {variable}',
        values: {
          variable: variable.name
        }
      }),
      color: "danger",
      onClick: () => setDeleteModalForVariable(id),
      "data-test-subj": "variablesRemoveButton"
    })
  }];
  return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_eui.EuiTitle, null, /*#__PURE__*/_react.default.createElement("h2", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "console.variablesPage.pageTitle",
    defaultMessage: "Variables"
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "s"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiText, {
    color: "subdued"
  }, /*#__PURE__*/_react.default.createElement("p", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "console.variablesPage.pageDescription",
    defaultMessage: "Define reusable placeholders for dynamic values in your queries."
  }))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "l"
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiBasicTable, {
    items: variables,
    columns: columns,
    itemId: "id",
    responsiveBreakpoint: false,
    className: "conVariablesTable",
    "data-test-subj": "variablesTable",
    itemIdToExpandedRowMap: itemIdToExpandedRowMap,
    noItemsMessage: _i18n.i18n.translate('console.variablesPage.table.noItemsMessage', {
      defaultMessage: 'No variables have been added yet'
    })
  }), isAddingVariable && /*#__PURE__*/_react.default.createElement(_variables_editor_form.VariableEditorForm, {
    onSubmit: onAddVariable,
    onCancel: () => setIsAddingVariable(false)
  }), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m"
  }), /*#__PURE__*/_react.default.createElement("div", null, /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
    "data-test-subj": "variablesAddButton",
    iconType: "plusInCircle",
    onClick: () => {
      setIsAddingVariable(true);
      collapseExpandedRows();
    },
    disabled: isAddingVariable
  }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "console.variablesPage.addButtonLabel",
    defaultMessage: "Add variable"
  }))), deleteModalForVariable && /*#__PURE__*/_react.default.createElement(_eui.EuiConfirmModal, {
    "aria-labelledby": deleteModalTitleId,
    title: _i18n.i18n.translate('console.variablesPage.deleteModal.title', {
      defaultMessage: 'Are you sure?'
    }),
    onCancel: () => setDeleteModalForVariable(null),
    onConfirm: () => deleteVariable(deleteModalForVariable),
    cancelButtonText: _i18n.i18n.translate('console.variablesPage.deleteModal.cancelButtonText', {
      defaultMessage: 'Cancel'
    }),
    confirmButtonText: _i18n.i18n.translate('console.variablesPage.deleteModal.confirmButtonText', {
      defaultMessage: 'Delete variable'
    }),
    buttonColor: "danger"
  }, /*#__PURE__*/_react.default.createElement("p", null, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
    id: "console.variablesPage.deleteModal.description",
    defaultMessage: "Deleting a variable is irreversible."
  }))));
};
exports.VariablesEditor = VariablesEditor;