"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getGraph = void 0;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getGraph = async (services, query) => {
  const {
    esClient,
    logger
  } = services;
  const {
    actorIds,
    eventIds,
    spaceId = 'default',
    start,
    end
  } = query;
  logger.trace(`Fetching graph for [eventIds: ${eventIds.join(', ')}] [actorIds: ${actorIds.join(', ')}] in [spaceId: ${spaceId}]`);
  const results = await fetchGraph({
    esClient,
    logger,
    start,
    end,
    eventIds,
    actorIds
  });

  // Convert results into set of nodes and edges
  const graphContext = parseRecords(logger, results.records);
  return {
    nodes: graphContext.nodes,
    edges: graphContext.edges
  };
};
exports.getGraph = getGraph;
const parseRecords = (logger, records) => {
  const nodesMap = {};
  const edgeLabelsNodes = {};
  const edgesMap = {};
  logger.trace(`Parsing records [length: ${records.length}]`);
  createNodes(logger, records, {
    nodesMap,
    edgeLabelsNodes
  });
  createEdgesAndGroups(logger, {
    edgeLabelsNodes,
    edgesMap,
    nodesMap
  });
  logger.trace(`Parsed [nodes: ${Object.keys(nodesMap).length}, edges: ${Object.keys(edgesMap).length}]`);

  // Sort groups to be first (fixes minor layout issue)
  const nodes = sortNodes(nodesMap);
  return {
    nodes,
    edges: Object.values(edgesMap)
  };
};
const fetchGraph = async ({
  esClient,
  logger,
  start,
  end,
  actorIds,
  eventIds
}) => {
  const query = `from logs-*
| WHERE event.action IS NOT NULL AND actor.entity.id IS NOT NULL
| EVAL isAlert = ${eventIds.length > 0 ? `event.id in (${eventIds.map((_id, idx) => `?al_id${idx}`).join(', ')})` : 'false'}
| STATS badge = COUNT(*),
  ips = VALUES(related.ip),
  // hosts = VALUES(related.hosts),
  users = VALUES(related.user)
    by actorIds = actor.entity.id,
      action = event.action,
      targetIds = target.entity.id,
      eventOutcome = event.outcome,
      isAlert
| LIMIT 1000`;
  logger.trace(`Executing query [${query}]`);
  return await esClient.asCurrentUser.helpers.esql({
    columnar: false,
    filter: {
      bool: {
        must: [{
          range: {
            '@timestamp': {
              gte: start,
              lte: end
            }
          }
        }, {
          bool: {
            should: [{
              terms: {
                'event.id': eventIds
              }
            }, {
              terms: {
                'actor.entity.id': actorIds
              }
            }],
            minimum_should_match: 1
          }
        }]
      }
    },
    query,
    // @ts-ignore - types are not up to date
    params: [...eventIds.map((id, idx) => ({
      [`al_id${idx}`]: id
    }))]
  }).toRecords();
};
const createNodes = (logger, records, context) => {
  const {
    nodesMap,
    edgeLabelsNodes
  } = context;
  for (const record of records) {
    const {
      ips,
      hosts,
      users,
      actorIds,
      action,
      targetIds,
      isAlert,
      eventOutcome
    } = record;
    const actorIdsArray = (0, _lodash.castArray)(actorIds);
    const targetIdsArray = (0, _lodash.castArray)(targetIds);
    logger.trace(`Parsing record [actorIds: ${actorIdsArray.join(', ')}, action: ${action}, targetIds: ${targetIdsArray.join(', ')}]`);

    // Create entity nodes
    [...actorIdsArray, ...targetIdsArray].forEach(id => {
      if (nodesMap[id] === undefined) {
        nodesMap[id] = {
          id,
          label: id,
          color: isAlert ? 'danger' : 'primary',
          ...determineEntityNodeShape(id, ips !== null && ips !== void 0 ? ips : [], hosts !== null && hosts !== void 0 ? hosts : [], users !== null && users !== void 0 ? users : [])
        };
        logger.trace(`Creating entity node [${id}]`);
      }
    });

    // Create label nodes
    for (const actorId of actorIdsArray) {
      for (const targetId of targetIdsArray) {
        const edgeId = `a(${actorId})-b(${targetId})`;
        if (edgeLabelsNodes[edgeId] === undefined) {
          edgeLabelsNodes[edgeId] = [];
        }
        const labelNode = {
          id: edgeId + `label(${action})outcome(${eventOutcome})`,
          label: action,
          source: actorId,
          target: targetId,
          color: isAlert ? 'danger' : eventOutcome === 'failed' ? 'warning' : 'primary',
          shape: 'label'
        };
        logger.trace(`Creating label node [${labelNode.id}]`);
        nodesMap[labelNode.id] = labelNode;
        edgeLabelsNodes[edgeId].push(labelNode.id);
      }
    }
  }
};
const determineEntityNodeShape = (actorId, ips, hosts, users) => {
  // If actor is a user return ellipse
  if (users.includes(actorId)) {
    return {
      shape: 'ellipse',
      icon: 'user'
    };
  }

  // If actor is a host return hexagon
  if (hosts.includes(actorId)) {
    return {
      shape: 'hexagon',
      icon: 'storage'
    };
  }

  // If actor is an IP return diamond
  if (ips.includes(actorId)) {
    return {
      shape: 'diamond',
      icon: 'globe'
    };
  }
  return {
    shape: 'hexagon',
    icon: 'questionInCircle'
  };
};
const sortNodes = nodesMap => {
  const groupNodes = [];
  const otherNodes = [];
  for (const node of Object.values(nodesMap)) {
    if (node.shape === 'group') {
      groupNodes.push(node);
    } else {
      otherNodes.push(node);
    }
  }
  return [...groupNodes, ...otherNodes];
};
const createEdgesAndGroups = (logger, context) => {
  const {
    edgeLabelsNodes,
    edgesMap,
    nodesMap
  } = context;
  Object.entries(edgeLabelsNodes).forEach(([edgeId, edgeLabelsIds]) => {
    // When there's more than one edge label, create a group node
    if (edgeLabelsIds.length === 1) {
      const edgeLabelId = edgeLabelsIds[0];
      connectEntitiesAndLabelNode(logger, edgesMap, nodesMap, nodesMap[edgeLabelId].source, edgeLabelId, nodesMap[edgeLabelId].target);
    } else {
      const groupNode = {
        id: `grp(${edgeId})`,
        shape: 'group'
      };
      nodesMap[groupNode.id] = groupNode;
      connectEntitiesAndLabelNode(logger, edgesMap, nodesMap, nodesMap[edgeLabelsIds[0]].source, groupNode.id, nodesMap[edgeLabelsIds[0]].target);
      edgeLabelsIds.forEach(edgeLabelId => {
        nodesMap[edgeLabelId].parentId = groupNode.id;
        connectEntitiesAndLabelNode(logger, edgesMap, nodesMap, groupNode.id, edgeLabelId, groupNode.id);
      });
    }
  });
};
const connectEntitiesAndLabelNode = (logger, edgesMap, nodesMap, sourceNodeId, labelNodeId, targetNodeId) => {
  [connectNodes(nodesMap, sourceNodeId, labelNodeId), connectNodes(nodesMap, labelNodeId, targetNodeId)].forEach(edge => {
    logger.trace(`Connecting nodes [${edge.source} -> ${edge.target}]`);
    edgesMap[edge.id] = edge;
  });
};
const connectNodes = (nodesMap, sourceNodeId, targetNodeId) => {
  const sourceNode = nodesMap[sourceNodeId];
  const targetNode = nodesMap[targetNodeId];
  const color = sourceNode.shape !== 'group' && targetNode.shape !== 'label' ? sourceNode.color : targetNode.shape !== 'group' ? targetNode.color : 'primary';
  return {
    id: `a(${sourceNodeId})-b(${targetNodeId})`,
    source: sourceNodeId,
    sourceShape: nodesMap[sourceNodeId].shape,
    target: targetNodeId,
    targetShape: nodesMap[targetNodeId].shape,
    color
  };
};