"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkCreate = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _domain = require("../../../common/types/domain");
var _runtime_types = require("../../common/runtime_types");
var _authorization = require("../../authorization");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _constants = require("../../common/constants");
var _api = require("../../../common/types/api");
var _validators = require("./validators");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkCreate = async (data, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      userActionService,
      licensingService,
      notificationService
    },
    user,
    logger,
    authorization: auth
  } = clientArgs;
  try {
    const decodedData = (0, _runtime_types.decodeWithExcessOrThrow)(_api.BulkCreateCasesRequestRt)(data);
    const configurations = await casesClient.configure.get();
    const customFieldsConfigurationMap = new Map(configurations.map(conf => [conf.owner, conf.customFields]));
    const casesWithIds = getCaseWithIds(decodedData);
    await auth.ensureAuthorized({
      operation: _authorization.Operations.createCase,
      entities: casesWithIds.map(theCase => ({
        owner: theCase.owner,
        id: theCase.id
      }))
    });
    const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
    const bulkCreateRequest = [];
    for (const theCase of casesWithIds) {
      const customFieldsConfiguration = customFieldsConfigurationMap.get(theCase.owner);
      validateRequest({
        theCase,
        customFieldsConfiguration,
        hasPlatinumLicenseOrGreater
      });
      bulkCreateRequest.push(createBulkCreateCaseRequest({
        theCase,
        user,
        customFieldsConfiguration
      }));
    }
    const bulkCreateResponse = await caseService.bulkCreateCases({
      cases: bulkCreateRequest,
      refresh: false
    });
    const userActions = [];
    const assigneesPerCase = [];
    const res = [];
    const [errors, casesSOs] = (0, _lodash.partition)(bulkCreateResponse.saved_objects, _error.isSOError);
    if (errors.length > 0) {
      const firstError = errors[0].error;
      if ((0, _error.isSODecoratedError)(firstError)) {
        throw new _boom.default.Boom(firstError.output.payload.error, {
          statusCode: firstError.output.statusCode,
          message: firstError.output.payload.message
        });
      }
      throw new _boom.default.Boom(firstError.error, {
        statusCode: firstError.statusCode,
        message: firstError.message
      });
    }
    for (const theCase of casesSOs) {
      userActions.push(createBulkCreateUserActionsRequest({
        theCase,
        user
      }));
      if (theCase.attributes.assignees && theCase.attributes.assignees.length !== 0) {
        const assigneesWithoutCurrentUser = theCase.attributes.assignees.filter(assignee => assignee.uid !== user.profile_uid);
        assigneesPerCase.push({
          assignees: assigneesWithoutCurrentUser,
          theCase
        });
      }
      res.push((0, _utils.flattenCaseSavedObject)({
        savedObject: theCase
      }));
    }
    await userActionService.creator.bulkCreateUserAction({
      userActions
    });
    if (assigneesPerCase.length > 0) {
      licensingService.notifyUsage(_constants.LICENSING_CASE_ASSIGNMENT_FEATURE);
      await notificationService.bulkNotifyAssignees(assigneesPerCase);
    }
    return (0, _runtime_types.decodeOrThrow)(_api.BulkCreateCasesResponseRt)({
      cases: res
    });
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to bulk create cases: ${error}`,
      error,
      logger
    });
  }
};
exports.bulkCreate = bulkCreate;
const getCaseWithIds = req => req.cases.map(theCase => {
  var _theCase$id;
  return {
    ...theCase,
    id: (_theCase$id = theCase.id) !== null && _theCase$id !== void 0 ? _theCase$id : _server.SavedObjectsUtils.generateId()
  };
});
const validateRequest = ({
  theCase,
  customFieldsConfiguration,
  hasPlatinumLicenseOrGreater
}) => {
  const customFieldsValidationParams = {
    requestCustomFields: theCase.customFields,
    customFieldsConfiguration
  };
  (0, _validators.validateCustomFields)(customFieldsValidationParams);
  validateAssigneesUsage({
    assignees: theCase.assignees,
    hasPlatinumLicenseOrGreater
  });
};
const validateAssigneesUsage = ({
  assignees,
  hasPlatinumLicenseOrGreater
}) => {
  /**
   * Assign users to a case is only available to Platinum+
   */

  if (assignees && assignees.length !== 0) {
    if (!hasPlatinumLicenseOrGreater) {
      throw _boom.default.forbidden('In order to assign users to cases, you must be subscribed to an Elastic Platinum license');
    }
  }
};
const createBulkCreateCaseRequest = ({
  theCase,
  customFieldsConfiguration,
  user
}) => {
  const {
    id,
    ...caseWithoutId
  } = theCase;

  /**
   * Trim title, category, description and tags
   * and fill out missing custom fields
   * before saving to ES
   */

  const normalizedCase = (0, _utils2.normalizeCreateCaseRequest)(caseWithoutId, customFieldsConfiguration);
  return {
    id,
    ...(0, _utils.transformNewCase)({
      user,
      newCase: normalizedCase
    })
  };
};
const createBulkCreateUserActionsRequest = ({
  theCase,
  user
}) => {
  var _theCase$attributes$s, _theCase$attributes$a, _theCase$attributes$c, _theCase$attributes$c2;
  const userActionPayload = {
    title: theCase.attributes.title,
    tags: theCase.attributes.tags,
    connector: theCase.attributes.connector,
    settings: theCase.attributes.settings,
    owner: theCase.attributes.owner,
    description: theCase.attributes.description,
    severity: (_theCase$attributes$s = theCase.attributes.severity) !== null && _theCase$attributes$s !== void 0 ? _theCase$attributes$s : _domain.CaseSeverity.LOW,
    assignees: (_theCase$attributes$a = theCase.attributes.assignees) !== null && _theCase$attributes$a !== void 0 ? _theCase$attributes$a : [],
    category: (_theCase$attributes$c = theCase.attributes.category) !== null && _theCase$attributes$c !== void 0 ? _theCase$attributes$c : null,
    customFields: (_theCase$attributes$c2 = theCase.attributes.customFields) !== null && _theCase$attributes$c2 !== void 0 ? _theCase$attributes$c2 : []
  };
  return {
    type: _domain.UserActionTypes.create_case,
    caseId: theCase.id,
    user,
    payload: userActionPayload,
    owner: theCase.attributes.owner
  };
};