"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCasesLocalStorage = void 0;
var _react = require("react");
var _use_cases_context = require("../components/cases_context/use_cases_context");
var _use_application = require("./lib/kibana/use_application");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useCasesLocalStorage = (key, initialValue) => {
  const isStorageInitialized = (0, _react.useRef)(false);
  const {
    appId
  } = (0, _use_application.useApplication)();
  const {
    owner
  } = (0, _use_cases_context.useCasesContext)();
  const lsKeyPrefix = owner.length > 0 ? owner.join('.') : appId;
  const lsKey = getLocalStorageKey(key, lsKeyPrefix);
  const [value, setValue] = (0, _react.useState)(() => getStorageItem(lsKey, initialValue));
  const setItem = (0, _react.useCallback)(newValue => {
    setValue(newValue);
    saveItemToStorage(lsKey, newValue);
  }, [lsKey]);
  if (!lsKeyPrefix) {
    return [initialValue, setItem];
  }
  if (lsKeyPrefix != null && !isStorageInitialized.current) {
    isStorageInitialized.current = true;
    setItem(getStorageItem(lsKey, initialValue));
  }
  return [value, setItem];
};
exports.useCasesLocalStorage = useCasesLocalStorage;
const getStorageItem = (key, initialValue) => {
  try {
    const value = localStorage.getItem(key);
    if (!value) {
      return initialValue;
    }
    return JSON.parse(value);
  } catch (error) {
    // silent errors
    return initialValue;
  }
};
const saveItemToStorage = (key, item) => {
  try {
    const value = JSON.stringify(item);
    localStorage.setItem(key, value);
  } catch (error) {
    // silent errors
  }
};
const getLocalStorageKey = (localStorageKey, prefix) => {
  return [prefix, localStorageKey].filter(Boolean).join('.');
};