"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.agentConfigurationRouteRepository = void 0;
var t = _interopRequireWildcard(require("io-ts"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _ioTsUtils = require("@kbn/io-ts-utils");
var _common = require("@kbn/observability-plugin/common");
var _create_or_update_configuration = require("./create_or_update_configuration");
var _search_configurations = require("./search_configurations");
var _find_exact_configuration = require("./find_exact_configuration");
var _list_configurations = require("./list_configurations");
var _get_environments = require("./get_environments");
var _delete_configuration = require("./delete_configuration");
var _create_apm_server_route = require("../../apm_routes/create_apm_server_route");
var _get_agent_name_by_service = require("./get_agent_name_by_service");
var _mark_applied_by_agent = require("./mark_applied_by_agent");
var _agent_configuration_intake_rt = require("../../../../common/agent_configuration/runtime_types/agent_configuration_intake_rt");
var _transactions = require("../../../lib/helpers/transactions");
var _sync_agent_configs_to_apm_package_policies = require("../../fleet/sync_agent_configs_to_apm_package_policies");
var _get_apm_event_client = require("../../../lib/helpers/get_apm_event_client");
var _create_internal_es_client = require("../../../lib/helpers/create_es_client/create_internal_es_client");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function throwNotFoundIfAgentConfigNotAvailable(featureFlags) {
  if (!featureFlags.agentConfigurationAvailable) {
    throw _boom.default.notFound();
  }
}

// get list of configurations
const agentConfigurationRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'GET /api/apm/settings/agent-configuration 2023-10-31',
  options: {
    tags: ['access:apm']
  },
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const apmIndices = await resources.getApmIndices();
    const [internalESClient, apmEventClient] = await Promise.all([(0, _create_internal_es_client.createInternalESClientWithResources)(resources), (0, _get_apm_event_client.getApmEventClient)(resources)]);
    const configurations = await (0, _list_configurations.listConfigurations)({
      internalESClient,
      apmIndices,
      apmEventClient
    });
    return {
      configurations
    };
  }
});

// get a single configuration
const getSingleAgentConfigurationRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'GET /api/apm/settings/agent-configuration/view 2023-10-31',
  params: t.partial({
    query: _agent_configuration_intake_rt.serviceRt
  }),
  options: {
    tags: ['access:apm']
  },
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const {
      params,
      logger
    } = resources;
    const {
      name,
      environment
    } = params.query;
    const service = {
      name,
      environment
    };
    const [internalESClient, apmEventClient] = await Promise.all([(0, _create_internal_es_client.createInternalESClientWithResources)(resources), (0, _get_apm_event_client.getApmEventClient)(resources)]);
    const exactConfig = await (0, _find_exact_configuration.findExactConfiguration)({
      service,
      internalESClient,
      apmEventClient
    });
    if (!exactConfig) {
      logger.info(`Config was not found for ${service.name}/${service.environment}`);
      throw _boom.default.notFound();
    }
    return exactConfig;
  }
});

// delete configuration
const deleteAgentConfigurationRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'DELETE /api/apm/settings/agent-configuration 2023-10-31',
  options: {
    tags: ['access:apm', 'access:apm_settings_write']
  },
  params: t.type({
    body: t.type({
      service: _agent_configuration_intake_rt.serviceRt
    })
  }),
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const {
      params,
      logger,
      core,
      telemetryUsageCounter
    } = resources;
    const {
      service
    } = params.body;
    const apmIndices = await resources.getApmIndices();
    const [internalESClient, apmEventClient] = await Promise.all([(0, _create_internal_es_client.createInternalESClientWithResources)(resources), (0, _get_apm_event_client.getApmEventClient)(resources)]);
    const exactConfig = await (0, _find_exact_configuration.findExactConfiguration)({
      service,
      internalESClient,
      apmEventClient
    });
    if (!exactConfig) {
      logger.info(`Config was not found for ${service.name}/${service.environment}`);
      throw _boom.default.notFound();
    }
    logger.info(`Deleting config ${service.name}/${service.environment} (${exactConfig.id})`);
    const deleteConfigurationResult = await (0, _delete_configuration.deleteConfiguration)({
      configurationId: exactConfig.id,
      internalESClient
    });
    if (resources.plugins.fleet) {
      await (0, _sync_agent_configs_to_apm_package_policies.syncAgentConfigsToApmPackagePolicies)({
        coreStartPromise: core.start(),
        fleetPluginStart: await resources.plugins.fleet.start(),
        internalESClient,
        apmIndices,
        telemetryUsageCounter
      });
      logger.info(`Updated Fleet integration policy for APM to remove the deleted agent configuration.`);
    }
    return deleteConfigurationResult;
  }
});

// create/update configuration
const createOrUpdateAgentConfigurationRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'PUT /api/apm/settings/agent-configuration 2023-10-31',
  options: {
    tags: ['access:apm', 'access:apm_settings_write']
  },
  params: t.intersection([t.partial({
    query: t.partial({
      overwrite: _ioTsUtils.toBooleanRt
    })
  }), t.type({
    body: _agent_configuration_intake_rt.agentConfigurationIntakeRt
  })]),
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const {
      params,
      logger,
      core,
      telemetryUsageCounter
    } = resources;
    const {
      body,
      query
    } = params;
    const apmIndices = await resources.getApmIndices();
    const [internalESClient, apmEventClient] = await Promise.all([(0, _create_internal_es_client.createInternalESClientWithResources)(resources), (0, _get_apm_event_client.getApmEventClient)(resources)]);

    // if the config already exists, it is fetched and updated
    // this is to avoid creating two configs with identical service params
    const exactConfig = await (0, _find_exact_configuration.findExactConfiguration)({
      service: body.service,
      internalESClient,
      apmEventClient
    });

    // if the config exists ?overwrite=true is required
    if (exactConfig && !query.overwrite) {
      throw _boom.default.badRequest(`A configuration already exists for "${body.service.name}/${body.service.environment}. Use ?overwrite=true to overwrite the existing configuration.`);
    }
    logger.info(`${exactConfig ? 'Updating' : 'Creating'} config ${body.service.name}/${body.service.environment}`);
    await (0, _create_or_update_configuration.createOrUpdateConfiguration)({
      configurationId: exactConfig === null || exactConfig === void 0 ? void 0 : exactConfig.id,
      configurationIntake: body,
      internalESClient
    });
    if (resources.plugins.fleet) {
      await (0, _sync_agent_configs_to_apm_package_policies.syncAgentConfigsToApmPackagePolicies)({
        coreStartPromise: core.start(),
        fleetPluginStart: await resources.plugins.fleet.start(),
        apmIndices,
        internalESClient,
        telemetryUsageCounter
      });
      logger.info(`Saved latest agent settings to Fleet integration policy for APM.`);
    }
  }
});
const searchParamsRt = t.intersection([t.type({
  service: _agent_configuration_intake_rt.serviceRt
}), t.partial({
  etag: t.string,
  mark_as_applied_by_agent: t.boolean
})]);
// Lookup single configuration (used by APM Server)
const agentConfigurationSearchRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'POST /api/apm/settings/agent-configuration/search 2023-10-31',
  params: t.type({
    body: searchParamsRt
  }),
  options: {
    tags: ['access:apm'],
    disableTelemetry: true
  },
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const {
      params,
      logger
    } = resources;
    const {
      service,
      etag,
      mark_as_applied_by_agent: markAsAppliedByAgent
    } = params.body;
    const internalESClient = await (0, _create_internal_es_client.createInternalESClientWithResources)(resources);
    const configuration = await (0, _search_configurations.searchConfigurations)({
      service,
      internalESClient
    });
    if (!configuration) {
      logger.debug(`[Central configuration] Config was not found for ${service.name}/${service.environment}`);
      return null;
    }

    // whether to update `applied_by_agent` field
    // It will be set to true of the etags match or if `markAsAppliedByAgent=true`
    // `markAsAppliedByAgent=true` means "force setting it to true regardless of etag". This is needed for Jaeger agent that doesn't have etags
    const willMarkAsApplied = (markAsAppliedByAgent || etag === configuration._source.etag) && !configuration._source.applied_by_agent;
    logger.debug(`[Central configuration] Config was found for:
        service.name = ${service.name},
        service.environment = ${service.environment},
        etag (requested) = ${etag},
        etag (existing) = ${configuration._source.etag},
        markAsAppliedByAgent = ${markAsAppliedByAgent},
        willMarkAsApplied = ${willMarkAsApplied}`);
    if (willMarkAsApplied) {
      await (0, _mark_applied_by_agent.markAppliedByAgent)({
        id: configuration._id,
        body: configuration._source,
        internalESClient
      });
    }
    return configuration;
  }
});

/*
 * Utility endpoints (not documented as part of the public API)
 */

// get environments for service
const listAgentConfigurationEnvironmentsRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'GET /api/apm/settings/agent-configuration/environments 2023-10-31',
  params: t.partial({
    query: t.partial({
      serviceName: t.string
    })
  }),
  options: {
    tags: ['access:apm']
  },
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const {
      context,
      params,
      config
    } = resources;
    const [internalESClient, apmEventClient] = await Promise.all([(0, _create_internal_es_client.createInternalESClientWithResources)(resources), (0, _get_apm_event_client.getApmEventClient)(resources)]);
    const coreContext = await context.core;
    const {
      serviceName
    } = params.query;
    const searchAggregatedTransactions = await (0, _transactions.getSearchTransactionsEvents)({
      apmEventClient,
      config,
      kuery: ''
    });
    const size = await coreContext.uiSettings.client.get(_common.maxSuggestions);
    const environments = await (0, _get_environments.getEnvironments)({
      serviceName,
      internalESClient,
      apmEventClient,
      searchAggregatedTransactions,
      size
    });
    return {
      environments
    };
  }
});

// get agentName for service
const agentConfigurationAgentNameRoute = (0, _create_apm_server_route.createApmServerRoute)({
  endpoint: 'GET /api/apm/settings/agent-configuration/agent_name 2023-10-31',
  params: t.type({
    query: t.type({
      serviceName: t.string
    })
  }),
  options: {
    tags: ['access:apm']
  },
  handler: async resources => {
    throwNotFoundIfAgentConfigNotAvailable(resources.featureFlags);
    const apmEventClient = await (0, _get_apm_event_client.getApmEventClient)(resources);
    const {
      params
    } = resources;
    const {
      serviceName
    } = params.query;
    const agentName = await (0, _get_agent_name_by_service.getAgentNameByService)({
      serviceName,
      apmEventClient
    });
    return {
      agentName
    };
  }
});
const agentConfigurationRouteRepository = exports.agentConfigurationRouteRepository = {
  ...agentConfigurationRoute,
  ...getSingleAgentConfigurationRoute,
  ...deleteAgentConfigurationRoute,
  ...createOrUpdateAgentConfigurationRoute,
  ...agentConfigurationSearchRoute,
  ...listAgentConfigurationEnvironmentsRoute,
  ...agentConfigurationAgentNameRoute
};