"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SystemActionScheduler = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lib = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SystemActionScheduler {
  constructor(context) {
    var _context$rule$systemA;
    (0, _defineProperty2.default)(this, "actions", []);
    this.context = context;
    const canGetSummarizedAlerts = !!context.ruleType.alerts && !!context.alertsClient.getSummarizedAlerts;

    // only process system actions when rule type supports summarized alerts
    this.actions = canGetSummarizedAlerts ? (_context$rule$systemA = context.rule.systemActions) !== null && _context$rule$systemA !== void 0 ? _context$rule$systemA : [] : [];
  }
  get priority() {
    return 1;
  }
  async getActionsToSchedule(_) {
    const executables = [];
    const results = [];
    for (const action of this.actions) {
      const options = {
        spaceId: this.context.taskInstance.params.spaceId,
        ruleId: this.context.rule.id,
        excludedAlertInstanceIds: this.context.rule.mutedInstanceIds,
        executionUuid: this.context.executionId
      };
      const summarizedAlerts = await (0, _lib.getSummarizedAlerts)({
        queryOptions: options,
        alertsClient: this.context.alertsClient
      });
      if (summarizedAlerts && summarizedAlerts.all.count !== 0) {
        executables.push({
          action,
          summarizedAlerts
        });
      }
    }
    if (executables.length === 0) return [];
    this.context.ruleRunMetricsStore.incrementNumberOfGeneratedActions(executables.length);
    const ruleUrl = (0, _lib.buildRuleUrl)({
      getViewInAppRelativeUrl: this.context.ruleType.getViewInAppRelativeUrl,
      kibanaBaseUrl: this.context.taskRunnerContext.kibanaBaseUrl,
      logger: this.context.logger,
      rule: this.context.rule,
      spaceId: this.context.taskInstance.params.spaceId
    });
    for (const {
      action,
      summarizedAlerts
    } of executables) {
      const {
        actionTypeId
      } = action;
      if (!(0, _lib.shouldScheduleAction)({
        action,
        actionsConfigMap: this.context.taskRunnerContext.actionsConfigMap,
        isActionExecutable: this.context.taskRunnerContext.actionsPlugin.isActionExecutable,
        logger: this.context.logger,
        ruleId: this.context.rule.id,
        ruleRunMetricsStore: this.context.ruleRunMetricsStore
      })) {
        continue;
      }
      const hasConnectorAdapter = this.context.taskRunnerContext.connectorAdapterRegistry.has(action.actionTypeId);

      // System actions without an adapter cannot be executed
      if (!hasConnectorAdapter) {
        this.context.logger.warn(`Rule "${this.context.rule.id}" skipped scheduling system action "${action.id}" because no connector adapter is configured`);
        continue;
      }
      this.context.ruleRunMetricsStore.incrementNumberOfTriggeredActions();
      this.context.ruleRunMetricsStore.incrementNumberOfTriggeredActionsByConnectorType(actionTypeId);
      const connectorAdapter = this.context.taskRunnerContext.connectorAdapterRegistry.get(action.actionTypeId);
      const connectorAdapterActionParams = connectorAdapter.buildActionParams({
        alerts: summarizedAlerts,
        rule: {
          id: this.context.rule.id,
          tags: this.context.rule.tags,
          name: this.context.rule.name,
          consumer: this.context.rule.consumer,
          producer: this.context.ruleType.producer
        },
        ruleUrl: ruleUrl === null || ruleUrl === void 0 ? void 0 : ruleUrl.absoluteUrl,
        spaceId: this.context.taskInstance.params.spaceId,
        params: action.params
      });
      const actionToRun = Object.assign(action, {
        params: connectorAdapterActionParams
      });
      results.push({
        actionToEnqueue: (0, _lib.formatActionToEnqueue)({
          action: actionToRun,
          apiKey: this.context.apiKey,
          executionId: this.context.executionId,
          ruleConsumer: this.context.ruleConsumer,
          ruleId: this.context.rule.id,
          ruleTypeId: this.context.ruleType.id,
          spaceId: this.context.taskInstance.params.spaceId
        }),
        actionToLog: {
          id: action.id,
          uuid: action.uuid,
          typeId: action.actionTypeId,
          alertSummary: {
            new: summarizedAlerts.new.count,
            ongoing: summarizedAlerts.ongoing.count,
            recovered: summarizedAlerts.recovered.count
          }
        }
      });
    }
    return results;
  }
}
exports.SystemActionScheduler = SystemActionScheduler;