"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SummaryActionScheduler = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _common = require("../../../../common");
var _lib = require("../lib");
var _inject_action_params = require("../../inject_action_params");
var _transform_action_params = require("../../transform_action_params");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SummaryActionScheduler {
  constructor(context) {
    var _context$rule$actions;
    (0, _defineProperty2.default)(this, "actions", []);
    this.context = context;
    const canGetSummarizedAlerts = !!context.ruleType.alerts && !!context.alertsClient.getSummarizedAlerts;

    // filter for summary actions where the rule type supports summarized alerts
    this.actions = (0, _lodash.compact)(((_context$rule$actions = context.rule.actions) !== null && _context$rule$actions !== void 0 ? _context$rule$actions : []).filter(action => (0, _lib.isSummaryAction)(action)).map(action => {
      if (!canGetSummarizedAlerts) {
        this.context.logger.error(`Skipping action "${action.id}" for rule "${this.context.rule.id}" because the rule type "${this.context.ruleType.name}" does not support alert-as-data.`);
        return null;
      }
      return action;
    }));
  }
  get priority() {
    return 0;
  }
  async getActionsToSchedule({
    activeCurrentAlerts,
    recoveredCurrentAlerts,
    throttledSummaryActions
  }) {
    const alerts = {
      ...activeCurrentAlerts,
      ...recoveredCurrentAlerts
    };
    const executables = [];
    const results = [];
    for (const action of this.actions) {
      if (
      // if summary action is throttled, we won't send any notifications
      !(0, _lib.isSummaryActionThrottled)({
        action,
        throttledSummaryActions,
        logger: this.context.logger
      })) {
        const actionHasThrottleInterval = (0, _lib.isActionOnInterval)(action);
        const optionsBase = {
          spaceId: this.context.taskInstance.params.spaceId,
          ruleId: this.context.rule.id,
          excludedAlertInstanceIds: this.context.rule.mutedInstanceIds,
          alertsFilter: action.alertsFilter
        };
        let options;
        if (actionHasThrottleInterval) {
          const throttleMills = (0, _common.parseDuration)(action.frequency.throttle);
          const start = new Date(Date.now() - throttleMills);
          options = {
            ...optionsBase,
            start,
            end: new Date()
          };
        } else {
          options = {
            ...optionsBase,
            executionUuid: this.context.executionId
          };
        }
        const summarizedAlerts = await (0, _lib.getSummarizedAlerts)({
          queryOptions: options,
          alertsClient: this.context.alertsClient
        });
        if (!actionHasThrottleInterval) {
          (0, _lib.logNumberOfFilteredAlerts)({
            logger: this.context.logger,
            numberOfAlerts: Object.entries(alerts).length,
            numberOfSummarizedAlerts: summarizedAlerts.all.count,
            action
          });
        }
        if (summarizedAlerts.all.count !== 0) {
          executables.push({
            action,
            summarizedAlerts
          });
        }
      }
    }
    if (executables.length === 0) return [];
    this.context.ruleRunMetricsStore.incrementNumberOfGeneratedActions(executables.length);
    for (const {
      action,
      summarizedAlerts
    } of executables) {
      const {
        actionTypeId
      } = action;
      if (!(0, _lib.shouldScheduleAction)({
        action,
        actionsConfigMap: this.context.taskRunnerContext.actionsConfigMap,
        isActionExecutable: this.context.taskRunnerContext.actionsPlugin.isActionExecutable,
        logger: this.context.logger,
        ruleId: this.context.rule.id,
        ruleRunMetricsStore: this.context.ruleRunMetricsStore
      })) {
        continue;
      }
      this.context.ruleRunMetricsStore.incrementNumberOfTriggeredActions();
      this.context.ruleRunMetricsStore.incrementNumberOfTriggeredActionsByConnectorType(actionTypeId);
      if ((0, _lib.isActionOnInterval)(action) && throttledSummaryActions) {
        throttledSummaryActions[action.uuid] = {
          date: new Date().toISOString()
        };
      }
      const {
        start,
        end
      } = (0, _lib.getSummaryActionTimeBounds)(action, this.context.rule.schedule, this.context.previousStartedAt);
      const ruleUrl = (0, _lib.buildRuleUrl)({
        end,
        getViewInAppRelativeUrl: this.context.ruleType.getViewInAppRelativeUrl,
        kibanaBaseUrl: this.context.taskRunnerContext.kibanaBaseUrl,
        logger: this.context.logger,
        rule: this.context.rule,
        spaceId: this.context.taskInstance.params.spaceId,
        start
      });
      const actionToRun = {
        ...action,
        params: (0, _inject_action_params.injectActionParams)({
          actionTypeId: action.actionTypeId,
          ruleUrl,
          ruleName: this.context.rule.name,
          actionParams: (0, _transform_action_params.transformSummaryActionParams)({
            alerts: summarizedAlerts,
            rule: this.context.rule,
            ruleTypeId: this.context.ruleType.id,
            actionId: action.id,
            actionParams: action.params,
            spaceId: this.context.taskInstance.params.spaceId,
            actionsPlugin: this.context.taskRunnerContext.actionsPlugin,
            actionTypeId: action.actionTypeId,
            kibanaBaseUrl: this.context.taskRunnerContext.kibanaBaseUrl,
            ruleUrl: ruleUrl === null || ruleUrl === void 0 ? void 0 : ruleUrl.absoluteUrl
          })
        })
      };
      results.push({
        actionToEnqueue: (0, _lib.formatActionToEnqueue)({
          action: actionToRun,
          apiKey: this.context.apiKey,
          executionId: this.context.executionId,
          ruleConsumer: this.context.ruleConsumer,
          ruleId: this.context.rule.id,
          ruleTypeId: this.context.ruleType.id,
          spaceId: this.context.taskInstance.params.spaceId
        }),
        actionToLog: {
          id: action.id,
          uuid: action.uuid,
          typeId: action.actionTypeId,
          alertSummary: {
            new: summarizedAlerts.new.count,
            ongoing: summarizedAlerts.ongoing.count,
            recovered: summarizedAlerts.recovered.count
          }
        }
      });
    }
    return results;
  }
}
exports.SummaryActionScheduler = SummaryActionScheduler;