"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.partiallyUpdateRule = partiallyUpdateRule;
exports.partiallyUpdateRuleWithEs = partiallyUpdateRuleWithEs;
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// We have calling code that references both RawRule and RuleAttributes,
// so we need to support both of these types (they are effectively the same)

// typed this way so we can send a SavedObjectClient or SavedObjectRepository

// direct, partial update to a rule saved object via scoped SavedObjectsClient
// using namespace set in the client
async function partiallyUpdateRule(savedObjectsClient, id, attributes, options = {}) {
  // ensure we only have the valid attributes that are not encrypted and are excluded from AAD
  const attributeUpdates = (0, _lodash.omit)(attributes, [..._.RuleAttributesToEncrypt, ..._.RuleAttributesIncludedInAAD]);
  const updateOptions = (0, _lodash.pick)(options, 'namespace', 'version', 'refresh');
  try {
    await savedObjectsClient.update(_.RULE_SAVED_OBJECT_TYPE, id, attributeUpdates, updateOptions);
  } catch (err) {
    if (options !== null && options !== void 0 && options.ignore404 && _server.SavedObjectsErrorHelpers.isNotFoundError(err)) {
      return;
    }
    throw err;
  }
}

// Explicit list of attributes that we allow to be partially updated
// There should be no overlap between this list and RuleAttributesIncludedInAAD or RuleAttributesToEncrypt
const RuleAttributesAllowedForPartialUpdate = ['executionStatus', 'lastRun', 'monitoring', 'nextRun', 'running', 'snoozeSchedule'];

// direct, partial update to a rule saved object via ElasticsearchClient

// we do this direct partial update to avoid the overhead of the SavedObjectsClient for
// only these allow-listed fields which don't impact encryption. in addition, because these
// fields are only updated by the system user at the end of a rule run, they should not
// need to be included in any (user-centric) audit logs.
async function partiallyUpdateRuleWithEs(esClient, id, attributes, options = {}) {
  // ensure we only have the valid attributes that are not encrypted and are excluded from AAD
  const attributeUpdates = (0, _lodash.omit)(attributes, [..._.RuleAttributesToEncrypt, ..._.RuleAttributesIncludedInAAD]);
  // ensure we only have attributes that we explicitly allow to be updated
  const attributesAllowedForUpdate = (0, _lodash.pick)(attributeUpdates, RuleAttributesAllowedForPartialUpdate);
  const updateParams = {
    id: `alert:${id}`,
    index: _coreSavedObjectsServer.ALERTING_CASES_SAVED_OBJECT_INDEX,
    ...(options.version ? (0, _coreSavedObjectsBaseServerInternal.decodeRequestVersion)(options.version) : {}),
    doc: {
      alert: attributesAllowedForUpdate
    },
    ...(options.refresh ? {
      refresh: options.refresh
    } : {})
  };
  if (options.ignore404) {
    await esClient.update(updateParams, {
      ignore: [404]
    });
  } else {
    await esClient.update(updateParams);
  }
}