"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActionErrorLog = getActionErrorLog;
exports.getActionErrorLogWithAuth = getActionErrorLogWithAuth;
var _lib = require("../../lib");
var _authorization = require("../../authorization");
var _audit_events = require("../common/audit_events");
var _format_execution_log_errors = require("../../lib/format_execution_log_errors");
var _common = require("../common");
var _get_rule = require("../../application/rule/methods/get/get_rule");
var _saved_objects = require("../../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const actionErrorLogDefaultFilter = 'event.provider:actions AND ((event.action:execute AND (event.outcome:failure OR kibana.alerting.status:warning)) OR (event.action:execute-timeout))';
async function getActionErrorLog(context, {
  id,
  dateStart,
  dateEnd,
  filter,
  page,
  perPage,
  sort
}) {
  var _context$auditLogger2;
  context.logger.debug(`getActionErrorLog(): getting action error logs for rule ${id}`);
  const rule = await (0, _get_rule.getRule)(context, {
    id,
    includeLegacyId: true
  });
  try {
    await context.authorization.ensureAuthorized({
      ruleTypeId: rule.alertTypeId,
      consumer: rule.consumer,
      operation: _authorization.ReadOperations.GetActionErrorLog,
      entity: _authorization.AlertingAuthorizationEntity.Rule
    });
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.GET_ACTION_ERROR_LOG,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id,
        name: rule.name
      },
      error
    }));
    throw error;
  }
  (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.GET_ACTION_ERROR_LOG,
    savedObject: {
      type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
      id,
      name: rule.name
    }
  }));

  // default duration of instance summary is 60 * rule interval
  const dateNow = new Date();
  const parsedDateStart = (0, _common.parseDate)(dateStart, 'dateStart', dateNow);
  const parsedDateEnd = (0, _common.parseDate)(dateEnd, 'dateEnd', dateNow);
  const eventLogClient = await context.getEventLogClient();
  try {
    const errorResult = await eventLogClient.findEventsBySavedObjectIds(_saved_objects.RULE_SAVED_OBJECT_TYPE, [id], {
      start: parsedDateStart.toISOString(),
      end: parsedDateEnd.toISOString(),
      page,
      per_page: perPage,
      filter: filter ? `(${actionErrorLogDefaultFilter}) AND (${filter})` : actionErrorLogDefaultFilter,
      sort: (0, _lib.convertEsSortToEventLogSort)(sort)
    }, rule.legacyId !== null ? [rule.legacyId] : undefined);
    return (0, _format_execution_log_errors.formatExecutionErrorsResult)(errorResult);
  } catch (err) {
    context.logger.debug(`rulesClient.getActionErrorLog(): error searching event log for rule ${id}: ${err.message}`);
    throw err;
  }
}
async function getActionErrorLogWithAuth(context, {
  id,
  dateStart,
  dateEnd,
  filter,
  page,
  perPage,
  sort,
  namespace
}) {
  var _context$auditLogger4;
  context.logger.debug(`getActionErrorLogWithAuth(): getting action error logs for rule ${id}`);
  let authorizationTuple;
  try {
    authorizationTuple = await context.authorization.getFindAuthorizationFilter(_authorization.AlertingAuthorizationEntity.Alert, {
      type: _authorization.AlertingAuthorizationFilterType.KQL,
      fieldNames: {
        ruleTypeId: 'kibana.alert.rule.rule_type_id',
        consumer: 'kibana.alert.rule.consumer'
      }
    });
  } catch (error) {
    var _context$auditLogger3;
    (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.GET_ACTION_ERROR_LOG,
      error
    }));
    throw error;
  }
  (_context$auditLogger4 = context.auditLogger) === null || _context$auditLogger4 === void 0 ? void 0 : _context$auditLogger4.log((0, _audit_events.ruleAuditEvent)({
    action: _audit_events.RuleAuditAction.GET_ACTION_ERROR_LOG,
    savedObject: {
      type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
      id
    }
  }));

  // default duration of instance summary is 60 * rule interval
  const dateNow = new Date();
  const parsedDateStart = (0, _common.parseDate)(dateStart, 'dateStart', dateNow);
  const parsedDateEnd = (0, _common.parseDate)(dateEnd, 'dateEnd', dateNow);
  const eventLogClient = await context.getEventLogClient();
  try {
    const errorResult = await eventLogClient.findEventsWithAuthFilter(_saved_objects.RULE_SAVED_OBJECT_TYPE, [id], authorizationTuple.filter, namespace, {
      start: parsedDateStart.toISOString(),
      end: parsedDateEnd.toISOString(),
      page,
      per_page: perPage,
      filter: filter ? `(${actionErrorLogDefaultFilter}) AND (${filter})` : actionErrorLogDefaultFilter,
      sort: (0, _lib.convertEsSortToEventLogSort)(sort)
    });
    return (0, _format_execution_log_errors.formatExecutionErrorsResult)(errorResult);
  } catch (err) {
    context.logger.debug(`rulesClient.getActionErrorLog(): error searching event log for rule ${id}: ${err.message}`);
    throw err;
  }
}