"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateRuleRoute = void 0;
var _update = require("../../../../../common/routes/rule/apis/update");
var _response = require("../../../../../common/routes/rule/response");
var _lib = require("../../../../lib");
var _types = require("../../../../types");
var _lib2 = require("../../../lib");
var _transforms = require("../../transforms");
var _validation = require("../../validation");
var _transforms2 = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateRuleRoute = (router, licenseState) => {
  router.put({
    path: `${_types.BASE_ALERTING_API_PATH}/rule/{id}`,
    options: {
      access: 'public',
      summary: `Update a rule`,
      tags: ['oas-tag:alerting']
    },
    validate: {
      request: {
        body: _update.updateBodySchemaV1,
        params: _update.updateParamsSchemaV1
      },
      response: {
        200: {
          body: () => _response.ruleResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        400: {
          description: 'Indicates an invalid schema or parameters.'
        },
        403: {
          description: 'Indicates that this call is forbidden.'
        },
        404: {
          description: 'Indicates a rule with the given ID does not exist.'
        },
        409: {
          description: 'Indicates that the rule has already been updated by another user.'
        }
      }
    }
  }, (0, _lib2.handleDisabledApiKeysError)(router.handleLegacyErrors((0, _lib2.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    const rulesClient = (await context.alerting).getRulesClient();
    const actionsClient = (await context.actions).getActionsClient();
    const rulesSettingsClient = (await context.alerting).getRulesSettingsClient(true);

    // Assert versioned inputs
    const updateRuleData = req.body;
    const updateRuleParams = req.params;
    try {
      /**
       * Throws an error if the group is not defined in default actions
       */
      const {
        actions: allActions = []
      } = updateRuleData;
      (0, _validation.validateRequiredGroupInDefaultActionsV1)({
        actions: allActions,
        isSystemAction: connectorId => actionsClient.isSystemAction(connectorId)
      });
      const actions = allActions.filter(action => !actionsClient.isSystemAction(action.id));
      const systemActions = allActions.filter(action => actionsClient.isSystemAction(action.id));
      const flappingSettings = await rulesSettingsClient.flapping().get();

      // TODO (http-versioning): Remove this cast, this enables us to move forward
      // without fixing all of other solution types
      const updatedRule = await rulesClient.update({
        id: updateRuleParams.id,
        data: (0, _transforms2.transformUpdateBodyV1)({
          updateBody: updateRuleData,
          actions,
          systemActions
        }),
        isFlappingEnabled: flappingSettings.enabled
      });

      // Assert versioned response type
      const response = {
        body: (0, _transforms.transformRuleToRuleResponseV1)(updatedRule)
      };
      return res.ok(response);
    } catch (e) {
      if (e instanceof _lib.RuleTypeDisabledError) {
        return e.sendResponse(res);
      }
      throw e;
    }
  }))));
};
exports.updateRuleRoute = updateRuleRoute;