"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateMonitoring = exports.transformRuleAttributesToRuleDomain = void 0;
var _lodash = require("lodash");
var _constants = require("../constants");
var _lib = require("../../../lib");
var _common = require("../../../rules_client/common");
var _is_rule_snoozed = require("../../../lib/is_rule_snoozed");
var _transform_raw_actions_to_domain_actions = require("./transform_raw_actions_to_domain_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INITIAL_LAST_RUN_METRICS = {
  duration: 0,
  total_search_duration_ms: null,
  total_indexing_duration_ms: null,
  total_alerts_detected: null,
  total_alerts_created: null,
  gap_duration_s: null
};
const transformEsExecutionStatus = (logger, ruleId, esRuleExecutionStatus) => {
  const {
    lastExecutionDate,
    lastDuration,
    status = _constants.ruleExecutionStatusValues.UNKNOWN,
    error,
    warning
  } = esRuleExecutionStatus;
  let parsedDateMillis = lastExecutionDate ? Date.parse(lastExecutionDate) : Date.now();
  if (isNaN(parsedDateMillis)) {
    logger.debug(`invalid ruleExecutionStatus lastExecutionDate "${lastExecutionDate}" in raw rule ${ruleId}`);
    parsedDateMillis = Date.now();
  }
  return {
    status,
    lastExecutionDate: new Date(parsedDateMillis),
    ...(lastDuration != null ? {
      lastDuration
    } : {}),
    ...(error ? {
      error
    } : {}),
    ...(warning ? {
      warning
    } : {})
  };
};
const updateMonitoring = ({
  monitoring,
  timestamp,
  duration
}) => {
  const {
    run,
    ...restMonitoring
  } = monitoring;
  const {
    last_run: lastRun,
    ...restRun
  } = run;
  const {
    metrics = INITIAL_LAST_RUN_METRICS
  } = lastRun;
  return {
    ...restMonitoring,
    run: {
      ...restRun,
      last_run: {
        timestamp,
        metrics: {
          ...metrics,
          duration
        }
      }
    }
  };
};
exports.updateMonitoring = updateMonitoring;
const transformEsMonitoring = (logger, ruleId, monitoring) => {
  if (!monitoring) {
    return undefined;
  }
  const lastRunDate = monitoring.run.last_run.timestamp;
  let parsedDateMillis = lastRunDate ? Date.parse(lastRunDate) : Date.now();
  if (isNaN(parsedDateMillis)) {
    logger.debug(`invalid monitoring last_run.timestamp "${lastRunDate}" in raw rule ${ruleId}`);
    parsedDateMillis = Date.now();
  }
  return updateMonitoring({
    monitoring,
    timestamp: new Date(parsedDateMillis).toISOString(),
    duration: monitoring.run.last_run.metrics.duration
  });
};
const transformRuleAttributesToRuleDomain = (esRule, transformParams, isSystemAction) => {
  var _getRuleSnoozeEndTime, _esRule$muteAll, _getActiveScheduledSn, _esRule$muteAll2;
  const {
    scheduledTaskId,
    executionStatus,
    monitoring,
    snoozeSchedule,
    lastRun
  } = esRule;
  const {
    id,
    logger,
    ruleType,
    references,
    includeSnoozeData = false,
    omitGeneratedValues = true
  } = transformParams;
  const snoozeScheduleDates = snoozeSchedule === null || snoozeSchedule === void 0 ? void 0 : snoozeSchedule.map(s => ({
    ...s,
    rRule: {
      ...s.rRule,
      dtstart: new Date(s.rRule.dtstart).toISOString(),
      ...(s.rRule.until ? {
        until: new Date(s.rRule.until).toISOString()
      } : {})
    }
  }));
  const includeSnoozeSchedule = snoozeSchedule !== undefined && !(0, _lodash.isEmpty)(snoozeSchedule);
  const isSnoozedUntil = includeSnoozeSchedule ? (_getRuleSnoozeEndTime = (0, _lib.getRuleSnoozeEndTime)({
    muteAll: (_esRule$muteAll = esRule.muteAll) !== null && _esRule$muteAll !== void 0 ? _esRule$muteAll : false,
    snoozeSchedule: snoozeSchedule
  })) === null || _getRuleSnoozeEndTime === void 0 ? void 0 : _getRuleSnoozeEndTime.toISOString() : null;
  const ruleDomainActions = (0, _transform_raw_actions_to_domain_actions.transformRawActionsToDomainActions)({
    ruleId: id,
    actions: esRule.actions,
    references,
    isSystemAction,
    omitGeneratedValues
  });
  const ruleDomainSystemActions = (0, _transform_raw_actions_to_domain_actions.transformRawActionsToDomainSystemActions)({
    ruleId: id,
    actions: esRule.actions,
    references,
    isSystemAction,
    omitGeneratedValues
  });
  const params = (0, _common.injectReferencesIntoParams)(id, ruleType, esRule.params, references || []);
  const activeSnoozes = (_getActiveScheduledSn = (0, _is_rule_snoozed.getActiveScheduledSnoozes)({
    snoozeSchedule: snoozeSchedule,
    muteAll: (_esRule$muteAll2 = esRule.muteAll) !== null && _esRule$muteAll2 !== void 0 ? _esRule$muteAll2 : false
  })) === null || _getActiveScheduledSn === void 0 ? void 0 : _getActiveScheduledSn.map(s => s.id);
  const rule = {
    id,
    enabled: esRule.enabled,
    name: esRule.name,
    tags: esRule.tags,
    alertTypeId: esRule.alertTypeId,
    consumer: esRule.consumer,
    schedule: esRule.schedule,
    actions: ruleDomainActions,
    systemActions: ruleDomainSystemActions,
    params,
    mapped_params: esRule.mapped_params,
    ...(scheduledTaskId ? {
      scheduledTaskId
    } : {}),
    createdBy: esRule.createdBy,
    updatedBy: esRule.updatedBy,
    createdAt: new Date(esRule.createdAt),
    updatedAt: new Date(esRule.updatedAt),
    apiKey: esRule.apiKey,
    apiKeyOwner: esRule.apiKeyOwner,
    apiKeyCreatedByUser: esRule.apiKeyCreatedByUser,
    throttle: esRule.throttle,
    muteAll: esRule.muteAll,
    notifyWhen: esRule.notifyWhen,
    mutedInstanceIds: esRule.mutedInstanceIds,
    ...(executionStatus ? {
      executionStatus: transformEsExecutionStatus(logger, id, executionStatus)
    } : {}),
    ...(monitoring ? {
      monitoring: transformEsMonitoring(logger, id, monitoring)
    } : {}),
    snoozeSchedule: snoozeScheduleDates !== null && snoozeScheduleDates !== void 0 ? snoozeScheduleDates : [],
    ...(includeSnoozeData ? {
      activeSnoozes,
      ...(isSnoozedUntil !== undefined ? {
        isSnoozedUntil: isSnoozedUntil ? new Date(isSnoozedUntil) : null
      } : {})
    } : {}),
    ...(lastRun ? {
      lastRun: {
        ...lastRun,
        ...(lastRun.outcomeMsg && !Array.isArray(lastRun.outcomeMsg) ? {
          outcomeMsg: lastRun.outcomeMsg ? [lastRun.outcomeMsg] : null
        } : {
          outcomeMsg: lastRun.outcomeMsg
        })
      }
    } : {}),
    ...(esRule.nextRun ? {
      nextRun: new Date(esRule.nextRun)
    } : {}),
    revision: esRule.revision,
    running: esRule.running,
    ...(esRule.alertDelay ? {
      alertDelay: esRule.alertDelay
    } : {}),
    ...(esRule.legacyId !== undefined ? {
      legacyId: esRule.legacyId
    } : {}),
    ...(esRule.flapping !== undefined ? {
      flapping: esRule.flapping
    } : {})
  };

  // Bad casts, but will fix once we fix all rule types
  const viewInAppRelativeUrl = ruleType.getViewInAppRelativeUrl && ruleType.getViewInAppRelativeUrl({
    rule: rule
  });
  if (viewInAppRelativeUrl) {
    rule.viewInAppRelativeUrl = viewInAppRelativeUrl;
  }

  // Remove all undefined keys to clean up the object

  for (const key in rule) {
    if (rule[key] === undefined) {
      delete rule[key];
    }
  }
  return rule;
};
exports.transformRuleAttributesToRuleDomain = transformRuleAttributesToRuleDomain;