"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Alert = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _lodash = require("lodash");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lib = require("../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class Alert {
  constructor(id, {
    state,
    meta = {}
  } = {}) {
    var _meta$uuid, _meta$maintenanceWind;
    (0, _defineProperty2.default)(this, "scheduledExecutionOptions", void 0);
    (0, _defineProperty2.default)(this, "meta", void 0);
    (0, _defineProperty2.default)(this, "state", void 0);
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "id", void 0);
    (0, _defineProperty2.default)(this, "alertAsData", void 0);
    this.id = id;
    this.state = state || {};
    this.context = {};
    this.meta = meta;
    this.meta.uuid = (_meta$uuid = meta.uuid) !== null && _meta$uuid !== void 0 ? _meta$uuid : (0, _uuid.v4)();
    this.meta.maintenanceWindowIds = (_meta$maintenanceWind = meta.maintenanceWindowIds) !== null && _meta$maintenanceWind !== void 0 ? _meta$maintenanceWind : [];
    if (!this.meta.flappingHistory) {
      this.meta.flappingHistory = [];
    }
  }
  getId() {
    return this.id;
  }
  getUuid() {
    return this.meta.uuid;
  }
  isAlertAsData() {
    return this.alertAsData !== undefined;
  }
  setAlertAsData(alertAsData) {
    this.alertAsData = alertAsData;
  }
  getAlertAsData() {
    return this.alertAsData;
  }
  getStart() {
    return this.state.start ? `${this.state.start}` : null;
  }
  hasScheduledActions() {
    return this.scheduledExecutionOptions !== undefined;
  }
  isThrottled({
    throttle,
    actionHash,
    uuid
  }) {
    if (this.scheduledExecutionOptions === undefined) {
      return false;
    }
    const throttleMills = throttle ? (0, _lib.parseDuration)(throttle) : 0;
    if (this.meta.lastScheduledActions && this.scheduledActionGroupIsUnchanged(this.meta.lastScheduledActions, this.scheduledExecutionOptions)) {
      if (uuid && actionHash) {
        if (this.meta.lastScheduledActions.actions) {
          const actionInState = this.meta.lastScheduledActions.actions[uuid] || this.meta.lastScheduledActions.actions[actionHash]; // actionHash must be removed once all the hash identifiers removed from the task state
          const lastTriggerDate = actionInState === null || actionInState === void 0 ? void 0 : actionInState.date;
          return !!(lastTriggerDate && new Date(lastTriggerDate).getTime() + throttleMills > Date.now());
        }
        return false;
      } else {
        return new Date(this.meta.lastScheduledActions.date).getTime() + throttleMills > Date.now();
      }
    }
    return false;
  }
  scheduledActionGroupHasChanged() {
    if (!this.meta.lastScheduledActions && this.scheduledExecutionOptions) {
      // it is considered a change when there are no previous scheduled actions
      // and new scheduled actions
      return true;
    }
    if (this.meta.lastScheduledActions && this.scheduledExecutionOptions) {
      // compare previous and new scheduled actions if both exist
      return !this.scheduledActionGroupIsUnchanged(this.meta.lastScheduledActions, this.scheduledExecutionOptions);
    }
    // no previous and no new scheduled actions
    return false;
  }
  scheduledActionGroupIsUnchanged(lastScheduledActions, scheduledExecutionOptions) {
    return lastScheduledActions.group === scheduledExecutionOptions.actionGroup;
  }
  getLastScheduledActions() {
    return this.meta.lastScheduledActions;
  }
  getScheduledActionOptions() {
    return this.scheduledExecutionOptions;
  }
  unscheduleActions() {
    this.scheduledExecutionOptions = undefined;
    return this;
  }
  getState() {
    return this.state;
  }
  getContext() {
    return this.context;
  }
  hasContext() {
    return !(0, _lodash.isEmpty)(this.context);
  }
  scheduleActions(actionGroup, context = {}) {
    this.ensureHasNoScheduledActions();
    this.setContext(context);
    this.scheduledExecutionOptions = {
      actionGroup,
      context,
      state: this.state
    };
    return this;
  }
  setContext(context) {
    this.context = context;
    return this;
  }
  ensureHasNoScheduledActions() {
    if (this.hasScheduledActions()) {
      throw new Error('Alert instance execution has already been scheduled, cannot schedule twice');
    }
  }
  replaceState(state) {
    this.state = state;
    return this;
  }
  updateLastScheduledActions(group, actionHash, uuid) {
    if (!this.meta.lastScheduledActions) {
      this.meta.lastScheduledActions = {};
    }
    const date = new Date().toISOString();
    this.meta.lastScheduledActions.group = group;
    this.meta.lastScheduledActions.date = date;
    if (this.meta.lastScheduledActions.group !== group) {
      this.meta.lastScheduledActions.actions = {};
    } else if (uuid) {
      if (!this.meta.lastScheduledActions.actions) {
        this.meta.lastScheduledActions.actions = {};
      }
      // remove deprecated actionHash
      if (!!actionHash && this.meta.lastScheduledActions.actions[actionHash]) {
        delete this.meta.lastScheduledActions.actions[actionHash];
      }
      this.meta.lastScheduledActions.actions[uuid] = {
        date
      };
    }
  }

  /**
   * Used to serialize alert instance state
   */
  toJSON() {
    return this.toRaw();
  }
  toRaw(recovered = false) {
    return recovered ? {
      // for a recovered alert, we only care to track the flappingHistory,
      // the flapping flag, and the UUID
      meta: {
        maintenanceWindowIds: this.meta.maintenanceWindowIds,
        flappingHistory: this.meta.flappingHistory,
        flapping: this.meta.flapping,
        uuid: this.meta.uuid,
        activeCount: this.meta.activeCount
      }
    } : {
      state: this.state,
      meta: this.meta
    };
  }
  setFlappingHistory(fh = []) {
    this.meta.flappingHistory = fh;
  }
  getFlappingHistory() {
    return this.meta.flappingHistory;
  }
  setFlapping(f) {
    this.meta.flapping = f;
  }
  getFlapping() {
    return this.meta.flapping || false;
  }
  incrementPendingRecoveredCount() {
    if (!this.meta.pendingRecoveredCount) {
      this.meta.pendingRecoveredCount = 0;
    }
    this.meta.pendingRecoveredCount++;
  }
  getPendingRecoveredCount() {
    return this.meta.pendingRecoveredCount || 0;
  }
  resetPendingRecoveredCount() {
    this.meta.pendingRecoveredCount = 0;
  }

  /**
   * Checks whether this alert exists in the given alert summary
   */
  isFilteredOut(summarizedAlerts) {
    if (summarizedAlerts === null) {
      return false;
    }

    // We check the alert UUID against both the alert ID and the UUID here
    // The framework generates a UUID for each new reported alert.
    // For lifecycle rule types, this UUID is written out in the ALERT_UUID field
    // so we can compare ALERT_UUID to getUuid()
    // For persistence rule types, the executor generates its own UUID which is a SHA
    // of the alert data and stores it in the ALERT_UUID and uses it as the alert ID
    // before reporting the alert back to the framework. The framework then generates
    // another UUID that is never persisted. For these alerts, we want to compare
    // ALERT_UUID to getId()
    //
    // Related issue: https://github.com/elastic/kibana/issues/144862

    return !summarizedAlerts.all.data.some(alert => {
      const alertUuid = (0, _lodash.get)(alert, _ruleDataUtils.ALERT_UUID);
      return alertUuid === this.getId() || alertUuid === this.getUuid();
    });
  }
  setMaintenanceWindowIds(maintenanceWindowIds = []) {
    this.meta.maintenanceWindowIds = maintenanceWindowIds;
  }
  getMaintenanceWindowIds() {
    var _this$meta$maintenanc;
    return (_this$meta$maintenanc = this.meta.maintenanceWindowIds) !== null && _this$meta$maintenanc !== void 0 ? _this$meta$maintenanc : [];
  }
  incrementActiveCount() {
    if (!this.meta.activeCount) {
      this.meta.activeCount = 0;
    }
    this.meta.activeCount++;
  }
  getActiveCount() {
    return this.meta.activeCount || 0;
  }
  resetActiveCount() {
    this.meta.activeCount = 0;
  }
}
exports.Alert = Alert;