"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setAllFieldCandidates = exports.logRateAnalysisFieldCandidatesSlice = exports.fetchFieldCandidates = void 0;
var _toolkit = require("@reduxjs/toolkit");
var _i18n = require("@kbn/i18n");
var _constants = require("@kbn/aiops-common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ecsIdentifiedMessage = _i18n.i18n.translate('xpack.aiops.logRateAnalysis.fieldCandidates.ecsIdentifiedMessage', {
  defaultMessage: 'The source documents were identified as ECS compliant.'
});
const fieldsDropdownHintMessage = _i18n.i18n.translate('xpack.aiops.logRateAnalysis.fieldCandidates.fieldsDropdownHintMessage', {
  defaultMessage: 'Use the "Fields" dropdown to adjust the selection.'
});
const getFieldSelectionMessage = (isECS, allItemsCount, selectedItemsCount) => {
  if (allItemsCount <= selectedItemsCount || selectedItemsCount < 2) return;
  const ecsMessage = isECS ? `${ecsIdentifiedMessage} ` : '';
  const fieldsSelectedMessage = _i18n.i18n.translate('xpack.aiops.logRateAnalysis.fieldCandidates.fieldsSelectedMessage', {
    defaultMessage: '{selectedItemsCount} out of {allItemsCount} fields were preselected for the analysis.',
    values: {
      selectedItemsCount,
      allItemsCount
    }
  });
  return `${ecsMessage}${fieldsSelectedMessage} ${fieldsDropdownHintMessage}`;
};
/**
 * Async thunk to fetch field candidates.
 */
const fetchFieldCandidates = exports.fetchFieldCandidates = (0, _toolkit.createAsyncThunk)('log_rate_analysis_field_candidates/fetch_field_candidates', async (options, thunkApi) => {
  const {
    http,
    abortCtrl,
    body,
    headers
  } = options;

  // Get field candidates so we're able to populate the field selection dropdown.
  const fieldCandidatesResp = await http.post(_constants.AIOPS_API_ENDPOINT.LOG_RATE_ANALYSIS_FIELD_CANDIDATES, {
    signal: abortCtrl.current.signal,
    version: '1',
    headers,
    ...(body && Object.keys(body).length > 0 ? {
      body: JSON.stringify(body)
    } : {})
  });
  const {
    isECS,
    keywordFieldCandidates,
    textFieldCandidates,
    selectedKeywordFieldCandidates,
    selectedTextFieldCandidates
  } = fieldCandidatesResp;
  const fieldFilterUniqueItems = [...keywordFieldCandidates, ...textFieldCandidates].sort();
  const fieldFilterUniqueSelectedItems = [...selectedKeywordFieldCandidates, ...selectedTextFieldCandidates];
  const fieldFilterSkippedItems = fieldFilterUniqueItems.filter(d => !fieldFilterUniqueSelectedItems.includes(d));
  thunkApi.dispatch(setAllFieldCandidates({
    fieldSelectionMessage: getFieldSelectionMessage(isECS, fieldFilterUniqueItems.length, fieldFilterUniqueSelectedItems.length),
    fieldFilterUniqueItems,
    fieldFilterSkippedItems,
    keywordFieldCandidates,
    textFieldCandidates,
    selectedKeywordFieldCandidates,
    selectedTextFieldCandidates
  }));
});
function getDefaultState() {
  return {
    isLoading: false,
    fieldFilterUniqueItems: [],
    fieldFilterSkippedItems: [],
    keywordFieldCandidates: [],
    textFieldCandidates: [],
    selectedKeywordFieldCandidates: [],
    selectedTextFieldCandidates: []
  };
}
const logRateAnalysisFieldCandidatesSlice = exports.logRateAnalysisFieldCandidatesSlice = (0, _toolkit.createSlice)({
  name: 'log_rate_analysis_field_candidates',
  initialState: getDefaultState(),
  reducers: {
    setAllFieldCandidates: (state, action) => {
      return {
        ...state,
        ...action.payload
      };
    }
  },
  extraReducers: builder => {
    builder.addCase(fetchFieldCandidates.pending, state => {
      state.isLoading = true;
    });
    builder.addCase(fetchFieldCandidates.fulfilled, state => {
      state.isLoading = false;
    });
  }
});

// Action creators are generated for each case reducer function
const {
  setAllFieldCandidates
} = logRateAnalysisFieldCandidatesSlice.actions;
exports.setAllFieldCandidates = setAllFieldCandidates;