"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTopTermRequest = exports.fetchTopTerms = void 0;
var _lodash = require("lodash");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _is_request_aborted_error = require("@kbn/aiops-common/is_request_aborted_error");
var _constants = require("../constants");
var _get_query_with_params = require("./get_query_with_params");
var _get_request_base = require("./get_request_base");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TODO Consolidate with duplicate `fetchDurationFieldCandidates` in
// `x-pack/plugins/observability_solution/apm/server/routes/correlations/queries/fetch_failed_events_correlation_p_values.ts`

const TOP_TERM_AGG_PREFIX = 'top_terms_';
const getTopTermRequest = (params, fieldNames, {
  wrap
}) => {
  var _params$timeFieldName;
  const query = (0, _get_query_with_params.getQueryWithParams)({
    params
  });
  const timeFieldName = (_params$timeFieldName = params.timeFieldName) !== null && _params$timeFieldName !== void 0 ? _params$timeFieldName : '@timestamp';
  let filter = [];
  if (query.bool && Array.isArray(query.bool.filter)) {
    filter = query.bool.filter.filter(d => Object.keys(d)[0] !== 'range');
    query.bool.filter = [...filter, {
      range: {
        [timeFieldName]: {
          gte: params.deviationMin,
          lt: params.deviationMax,
          format: 'epoch_millis'
        }
      }
    }];
  }
  const termAggs = fieldNames.reduce((aggs, fieldName, index) => {
    aggs[`${TOP_TERM_AGG_PREFIX}${index}`] = {
      terms: {
        field: fieldName,
        size: _constants.LOG_RATE_ANALYSIS_SETTINGS.TOP_TERMS_FALLBACK_SIZE
      }
    };
    return aggs;
  }, {});
  const body = {
    query,
    size: 0,
    aggs: wrap(termAggs)
  };
  return {
    ...(0, _get_request_base.getRequestBase)(params),
    body
  };
};
exports.getTopTermRequest = getTopTermRequest;
const fetchTopTerms = async ({
  esClient,
  abortSignal,
  emitError,
  logger,
  arguments: args
}) => {
  // The default value of 1 means no sampling will be used
  const {
    fieldNames,
    sampleProbability = 1,
    ...params
  } = args;
  const randomSamplerWrapper = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
    probability: sampleProbability,
    seed: _constants.RANDOM_SAMPLER_SEED
  });
  const result = [];
  const resp = await esClient.search(getTopTermRequest(params, fieldNames, randomSamplerWrapper), {
    signal: abortSignal,
    maxRetries: 0
  });
  if (resp.aggregations === undefined) {
    if (!(0, _is_request_aborted_error.isRequestAbortedError)(resp)) {
      if (logger) {
        logger.error(`Failed to fetch terms aggregation for field names ${fieldNames.join()}, got: \n${JSON.stringify(resp, null, 2)}`);
      }
      if (emitError) {
        emitError(`Failed to fetch terms aggregation for field names ${fieldNames.join()}.`);
      }
    }
    return result;
  }
  const unwrappedResp = randomSamplerWrapper.unwrap(resp.aggregations);
  for (const [index, fieldName] of fieldNames.entries()) {
    const termBuckets = unwrappedResp[`${TOP_TERM_AGG_PREFIX}${index}`];
    for (const bucket of termBuckets.buckets) {
      result.push({
        key: `${fieldName}:${String(bucket.key)}`,
        type: _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD,
        fieldName,
        fieldValue: String(bucket.key),
        doc_count: bucket.doc_count,
        bg_count: 0,
        total_doc_count: 0,
        total_bg_count: 0,
        score: 0,
        pValue: 1,
        normalizedScore: 0
      });
    }
  }
  return (0, _lodash.uniqBy)(result, d => `${d.fieldName},${d.fieldValue}`);
};
exports.fetchTopTerms = fetchTopTerms;