"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeConnectorRoute = void 0;
var _common = require("../../../../common");
var _action_execution_source = require("../../../lib/action_execution_source");
var _verify_access_and_context = require("../../verify_access_and_context");
var _response = require("../../../../common/routes/connector/response");
var _execute = require("../../../../common/routes/connector/apis/execute");
var _transforms = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const executeConnectorRoute = (router, licenseState) => {
  router.post({
    path: `${_common.BASE_ACTION_API_PATH}/connector/{id}/_execute`,
    options: {
      access: 'public',
      summary: `Run a connector`,
      description: 'You can use this API to test an action that involves interaction with Kibana services or integrations with third-party systems.',
      tags: ['oas-tag:connectors']
    },
    validate: {
      request: {
        body: _execute.executeConnectorRequestBodySchemaV1,
        params: _execute.executeConnectorRequestParamsSchemaV1
      },
      response: {
        200: {
          description: 'Indicates a successful call.',
          body: () => _response.connectorResponseSchemaV1
        }
      }
    }
  }, router.handleLegacyErrors((0, _verify_access_and_context.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    const actionsClient = (await context.actions).getActionsClient();
    const {
      params
    } = req.body;
    const {
      id
    } = req.params;
    if (actionsClient.isSystemAction(id)) {
      return res.badRequest({
        body: 'Execution of system action is not allowed'
      });
    }
    const body = await actionsClient.execute({
      params,
      actionId: id,
      source: (0, _action_execution_source.asHttpRequestExecutionSource)(req),
      relatedSavedObjects: []
    });
    return body ? res.ok({
      body: (0, _transforms.transformExecuteConnectorResponseV1)(body)
    }) : res.noContent();
  })));
};
exports.executeConnectorRoute = executeConnectorRoute;