"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ESQLAstAdapter = void 0;
var _esqlValidationAutocomplete = require("@kbn/esql-validation-autocomplete");
var _positions = require("./converters/positions");
var _hover = require("./hover/hover");
var _utils = require("./shared/utils");
var _signature = require("./signature");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class ESQLAstAdapter {
  constructor(worker, callbacks) {
    this.worker = worker;
    this.callbacks = callbacks;
  }
  async getAstWorker(model) {
    const worker = await this.worker(model.uri);
    return worker.getAst;
  }
  async getAst(model, code) {
    const getAstFn = await this.getAstWorker(model);
    return getAstFn(code !== null && code !== void 0 ? code : model.getValue());
  }
  async validate(model, code) {
    const getAstFn = await this.getAstWorker(model);
    const text = code !== null && code !== void 0 ? code : model.getValue();
    const {
      errors,
      warnings
    } = await (0, _esqlValidationAutocomplete.validateQuery)(text, getAstFn, undefined, this.callbacks);
    const monacoErrors = (0, _positions.wrapAsMonacoMessages)(text, errors);
    const monacoWarnings = (0, _positions.wrapAsMonacoMessages)(text, warnings);
    return {
      errors: monacoErrors,
      warnings: monacoWarnings
    };
  }
  async suggestSignature(model, position, context) {
    const getAstFn = await this.getAstWorker(model);
    return (0, _signature.getSignatureHelp)(model, position, context, getAstFn);
  }
  async getHover(model, position, token) {
    const getAstFn = await this.getAstWorker(model);
    return (0, _hover.getHoverItem)(model, position, token, getAstFn, this.callbacks);
  }
  async autocomplete(model, position, context) {
    const getAstFn = await this.getAstWorker(model);
    const fullText = model.getValue();
    const offset = (0, _utils.monacoPositionToOffset)(fullText, position);
    const suggestionEntries = await (0, _esqlValidationAutocomplete.suggest)(fullText, offset, context, getAstFn, this.callbacks);
    return {
      suggestions: suggestionEntries.map(suggestion => ({
        ...suggestion,
        range: undefined
      }))
    };
  }
  async codeAction(model, range, context) {
    const getAstFn = await this.getAstWorker(model);
    const codeActions = await (0, _esqlValidationAutocomplete.getActions)(model.getValue(), context.markers, getAstFn, undefined, this.callbacks);
    return codeActions;
  }
}
exports.ESQLAstAdapter = ESQLAstAdapter;