"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupCapabilitiesSwitcher = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _license = require("../../../common/license");
var _capabilities = require("../../../common/types/capabilities");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const setupCapabilitiesSwitcher = (coreSetup, license$, enabledFeatures, logger) => {
  coreSetup.capabilities.registerSwitcher(getSwitcher(license$, logger, enabledFeatures), {
    capabilityPath: 'ml.*'
  });
};
exports.setupCapabilitiesSwitcher = setupCapabilitiesSwitcher;
function getSwitcher(license$, logger, enabledFeatures) {
  return async (request, capabilities) => {
    const isAnonymousRequest = !request.route.options.authRequired;
    if (isAnonymousRequest) {
      return {};
    }
    try {
      const license = await (0, _rxjs.firstValueFrom)(license$);
      const mlEnabled = (0, _license.isMlEnabled)(license);
      const originalCapabilities = capabilities.ml;
      const mlCaps = (0, _lodash.cloneDeep)(originalCapabilities);

      // full license, leave capabilities as they were
      if (mlEnabled && (0, _license.isFullLicense)(license)) {
        return {
          ml: applyEnabledFeatures(mlCaps, enabledFeatures)
        };
      }

      // not full license, switch off all capabilities
      Object.keys(mlCaps).forEach(k => {
        mlCaps[k] = false;
      });

      // for a basic license, reapply the original capabilities for the basic license features
      if (mlEnabled && (0, _license.isMinimumLicense)(license)) {
        _capabilities.basicLicenseMlCapabilities.forEach(c => mlCaps[c] = originalCapabilities[c]);
      }
      return {
        ml: mlCaps
      };
    } catch (e) {
      logger.debug(`Error updating capabilities for ML based on licensing: ${e}`);
      return {};
    }
  };
}
function applyEnabledFeatures(mlCaps, {
  ad,
  dfa,
  nlp
}) {
  mlCaps.isADEnabled = ad;
  mlCaps.isDFAEnabled = dfa;
  mlCaps.isNLPEnabled = nlp;
  mlCaps.canViewMlNodes = mlCaps.canViewMlNodes && ad && dfa && nlp;
  if (ad === false) {
    for (const c of _capabilities.featureCapabilities.ad) {
      mlCaps[c] = false;
    }
  }
  if (dfa === false) {
    for (const c of _capabilities.featureCapabilities.dfa) {
      mlCaps[c] = false;
    }
  }
  if (nlp === false && dfa === false) {
    for (const c of _capabilities.featureCapabilities.nlp) {
      mlCaps[c] = false;
    }
  }
  return mlCaps;
}