"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.trainedModelsApiProvider = trainedModelsApiProvider;
exports.useTrainedModelsApiService = useTrainedModelsApiService;
var _react = require("react");
var _app = require("../../../../common/constants/app");
var _kibana = require("../../contexts/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Service with APIs calls to perform operations with trained models.
 * @param httpService
 */
function trainedModelsApiProvider(httpService) {
  return {
    /**
     * Fetches the trained models list available for download.
     */
    getTrainedModelDownloads() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/model_downloads`,
        method: 'GET',
        version: '1'
      });
    },
    /**
     * Gets ELSER config for download based on the cluster OS and CPU architecture.
     */
    getElserConfig(options) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/elser_config`,
        method: 'GET',
        ...(options ? {
          query: options
        } : {}),
        version: '1'
      });
    },
    /**
     * Fetches configuration information for a trained inference model.
     * @param modelId - Model ID, collection of Model IDs or Model ID pattern.
     *                  Fetches all In case nothing is provided.
     * @param params - Optional query params
     */
    getTrainedModels(modelId, params) {
      const model = Array.isArray(modelId) ? modelId.join(',') : modelId;
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models${model ? `/${model}` : ''}`,
        method: 'GET',
        ...(params ? {
          query: params
        } : {}),
        version: '1'
      });
    },
    /**
     * Fetches usage information for trained inference models.
     * @param modelId - Model ID, collection of Model IDs or Model ID pattern.
     *                  Fetches all In case nothing is provided.
     * @param params - Optional query params
     */
    getTrainedModelStats(modelId, params) {
      const model = Array.isArray(modelId) ? modelId.join(',') : modelId;
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models${model ? `/${model}` : ''}/_stats`,
        method: 'GET',
        version: '1'
      });
    },
    /**
     * Fetches pipelines associated with provided models
     * @param modelId - Model ID, collection of Model IDs.
     */
    getTrainedModelPipelines(modelId) {
      let model = modelId;
      if (Array.isArray(modelId)) {
        model = modelId.join(',');
      }
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/${model}/pipelines`,
        method: 'GET',
        version: '1'
      });
    },
    /**
     * Fetches all ingest pipelines
     */
    getAllIngestPipelines() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/ingest_pipelines`,
        method: 'GET',
        version: '1'
      });
    },
    /**
     * Creates inference pipeline
     */
    createInferencePipeline(pipelineName, pipeline) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/create_inference_pipeline`,
        method: 'POST',
        body: JSON.stringify({
          pipeline,
          pipelineName
        }),
        version: '1'
      });
    },
    /**
     * Deletes an existing trained inference model.
     * @param modelId - Model ID
     */
    deleteTrainedModel(modelId, options = {
      with_pipelines: false,
      force: false
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/${modelId}`,
        method: 'DELETE',
        version: '1',
        query: options
      });
    },
    /**
     * Gets model config based on the cluster OS and CPU architecture.
     */
    getCuratedModelConfig(modelName, options) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/curated_model_config/${modelName}`,
        method: 'GET',
        ...(options ? {
          query: options
        } : {}),
        version: '1'
      });
    },
    getTrainedModelsNodesOverview() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/model_management/nodes_overview`,
        method: 'GET',
        version: '1'
      });
    },
    startModelAllocation(modelId, queryParams) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/${modelId}/deployment/_start`,
        method: 'POST',
        query: queryParams,
        version: '1'
      });
    },
    stopModelAllocation(modelId, deploymentsIds, options = {
      force: false
    }) {
      const force = options === null || options === void 0 ? void 0 : options.force;
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/${modelId}/${deploymentsIds.join(',')}/deployment/_stop`,
        method: 'POST',
        query: {
          force
        },
        version: '1'
      });
    },
    updateModelDeployment(modelId, deploymentId, params) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/${modelId}/${deploymentId}/deployment/_update`,
        method: 'POST',
        body: JSON.stringify(params),
        version: '1'
      });
    },
    inferTrainedModel(modelId, deploymentsId, payload, timeout) {
      const body = JSON.stringify(payload);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/infer/${modelId}/${deploymentsId}`,
        method: 'POST',
        body,
        ...(timeout ? {
          query: {
            timeout
          }
        } : {}),
        version: '1'
      });
    },
    trainedModelPipelineSimulate(pipeline, docs) {
      const body = JSON.stringify({
        pipeline,
        docs
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/pipeline_simulate`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    memoryUsage(type, node, showClosedJobs = false) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/model_management/memory_usage`,
        method: 'GET',
        query: {
          type,
          node,
          showClosedJobs
        },
        version: '1'
      });
    },
    putTrainedModelConfig(modelId, config) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/${modelId}`,
        method: 'PUT',
        body: JSON.stringify(config),
        version: '1'
      });
    },
    installElasticTrainedModelConfig(modelId) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/install_elastic_trained_model/${modelId}`,
        method: 'POST',
        version: '1'
      });
    },
    getModelsDownloadStatus() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/trained_models/download_status`,
        method: 'GET',
        version: '1'
      });
    }
  };
}
/**
 * Hooks for accessing {@link TrainedModelsApiService} in React components.
 */
function useTrainedModelsApiService() {
  const {
    services: {
      mlServices: {
        httpService
      }
    }
  } = (0, _kibana.useMlKibana)();
  return (0, _react.useMemo)(() => trainedModelsApiProvider(httpService), [httpService]);
}