"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.actionsMenuContent = actionsMenuContent;
var _check_capabilities = require("../../../../capabilities/check_capabilities");
var _check_ml_nodes = require("../../../../ml_nodes_check/check_ml_nodes");
var _job_actions = require("../../../../../../common/constants/job_actions");
var _utils = require("../utils");
var _i18n = require("@kbn/i18n");
var _jobs_utils = require("../../../jobs_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function actionsMenuContent(showEditJobFlyout, showDatafeedChartFlyout, showDeleteJobModal, showResetJobModal, showStartDatafeedModal, showCloseJobsConfirmModal, showStopDatafeedsConfirmModal, refreshJobs, showCreateAlertFlyout) {
  const canCreateJob = (0, _check_capabilities.checkPermission)('canCreateJob') && (0, _check_ml_nodes.mlNodesAvailable)();
  const canUpdateJob = (0, _check_capabilities.checkPermission)('canUpdateJob');
  const canDeleteJob = (0, _check_capabilities.checkPermission)('canDeleteJob');
  const canGetDatafeeds = (0, _check_capabilities.checkPermission)('canGetDatafeeds');
  const canUpdateDatafeed = (0, _check_capabilities.checkPermission)('canUpdateDatafeed');
  const canStartStopDatafeed = (0, _check_capabilities.checkPermission)('canStartStopDatafeed') && (0, _check_ml_nodes.mlNodesAvailable)();
  const canCloseJob = (0, _check_capabilities.checkPermission)('canCloseJob') && (0, _check_ml_nodes.mlNodesAvailable)();
  const canResetJob = (0, _check_capabilities.checkPermission)('canResetJob') && (0, _check_ml_nodes.mlNodesAvailable)();
  const canCreateMlAlerts = (0, _check_capabilities.checkPermission)('canCreateMlAlerts');
  return [{
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.startDatafeedLabel', {
      defaultMessage: 'Start datafeed'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.startDatafeedDescription', {
      defaultMessage: 'Start datafeed'
    }),
    icon: 'play',
    enabled: item => isJobBlocked(item) === false && canStartStopDatafeed,
    available: item => (0, _utils.isStartable)([item]),
    onClick: item => {
      showStartDatafeedModal([item]);
      closeMenu();
    },
    'data-test-subj': 'mlActionButtonStartDatafeed'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.stopDatafeedLabel', {
      defaultMessage: 'Stop datafeed'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.stopDatafeedDescription', {
      defaultMessage: 'Stop datafeed'
    }),
    icon: 'stop',
    enabled: item => isJobBlocked(item) === false && canStartStopDatafeed,
    available: item => (0, _utils.isStoppable)([item]),
    onClick: item => {
      if ((0, _jobs_utils.isManagedJob)(item)) {
        showStopDatafeedsConfirmModal([item]);
      } else {
        (0, _utils.stopDatafeeds)([item], refreshJobs);
      }
      closeMenu(true);
    },
    'data-test-subj': 'mlActionButtonStopDatafeed'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.createAlertLabel', {
      defaultMessage: 'Create alert rule'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.createAlertLabel', {
      defaultMessage: 'Create alert rule'
    }),
    icon: 'bell',
    enabled: item => isJobBlocked(item) === false,
    available: () => canCreateMlAlerts,
    onClick: item => {
      showCreateAlertFlyout([item.id]);
      closeMenu(true);
    },
    'data-test-subj': 'mlActionButtonCreateAlert'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.closeJobLabel', {
      defaultMessage: 'Close job'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.closeJobDescription', {
      defaultMessage: 'Close job'
    }),
    icon: 'cross',
    enabled: item => isJobBlocked(item) === false && canCloseJob,
    available: item => (0, _utils.isClosable)([item]),
    onClick: item => {
      if ((0, _jobs_utils.isManagedJob)(item)) {
        showCloseJobsConfirmModal([item]);
      } else {
        (0, _utils.closeJobs)([item], refreshJobs);
      }
      closeMenu(true);
    },
    'data-test-subj': 'mlActionButtonCloseJob'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.resetJobLabel', {
      defaultMessage: 'Reset job'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.resetJobDescription', {
      defaultMessage: 'Reset job'
    }),
    icon: 'refresh',
    enabled: item => isResetEnabled(item) && canResetJob,
    available: item => (0, _utils.isResettable)([item]),
    onClick: item => {
      showResetJobModal([item]);
      closeMenu(true);
    },
    'data-test-subj': 'mlActionButtonResetJob'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.cloneJobLabel', {
      defaultMessage: 'Clone job'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.cloneJobDescription', {
      defaultMessage: 'Clone job'
    }),
    icon: 'copy',
    enabled: item => {
      // We only allow cloning of a job if the user has the right permissions and can still access
      // the indexPattern the job was created for. An indexPattern could either have been deleted
      // since the the job was created or the current user doesn't have the required permissions to
      // access the indexPattern.
      return isJobBlocked(item) === false && canCreateJob;
    },
    onClick: item => {
      (0, _utils.cloneJob)(item.id);
      closeMenu(true);
    },
    'data-test-subj': 'mlActionButtonCloneJob'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.viewDatafeedCountsLabel', {
      defaultMessage: 'View datafeed counts'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.viewDatafeedCountsDescription', {
      defaultMessage: 'View datafeed counts'
    }),
    icon: 'visAreaStacked',
    enabled: () => canGetDatafeeds,
    available: () => canGetDatafeeds,
    onClick: item => {
      showDatafeedChartFlyout(item);
      closeMenu();
    },
    'data-test-subj': 'mlActionButtonViewDatafeedChart'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.editJobLabel', {
      defaultMessage: 'Edit job'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.editJobDescription', {
      defaultMessage: 'Edit job'
    }),
    icon: 'pencil',
    enabled: item => isJobBlocked(item) === false && canUpdateJob && canUpdateDatafeed,
    onClick: item => {
      showEditJobFlyout(item);
      closeMenu();
    },
    'data-test-subj': 'mlActionButtonEditJob'
  }, {
    name: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.deleteJobLabel', {
      defaultMessage: 'Delete job'
    }),
    description: _i18n.i18n.translate('xpack.ml.jobsList.managementActions.deleteJobDescription', {
      defaultMessage: 'Delete job'
    }),
    icon: 'trash',
    color: 'danger',
    enabled: () => canDeleteJob,
    onClick: item => {
      showDeleteJobModal([item]);
      closeMenu();
    },
    'data-test-subj': 'mlActionButtonDeleteJob'
  }];
}
function isResetEnabled(item) {
  if (item.blocked === undefined || item.blocked.reason === _job_actions.JOB_ACTION.RESET) {
    return true;
  }
  return false;
}
function isJobBlocked(item) {
  return item.blocked !== undefined;
}
function closeMenu(now = false) {
  if (now) {
    document.querySelector('.euiTable').click();
  } else {
    window.setTimeout(() => {
      const modalBody = document.querySelector('.euiModalBody');
      if (modalBody) {
        modalBody.click();
      } else {
        document.querySelector('.euiTable').click();
      }
    }, 500);
  }
}