"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.checkIndexExists = checkIndexExists;
exports.createKibanaDataView = void 0;
exports.retryIndexExistsCheck = retryIndexExistsCheck;
var _i18n = require("@kbn/i18n");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _public = require("@kbn/data-plugin/public");
var _ml_api_service = require("../../services/ml_api_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function delay(ms = 1000) {
  return new Promise(resolve => {
    setTimeout(resolve, ms);
  });
}
async function checkIndexExists(destIndex) {
  let resp;
  let errorMessage;
  try {
    resp = await _ml_api_service.ml.checkIndicesExists({
      indices: [destIndex]
    });
  } catch (e) {
    errorMessage = (0, _mlErrorUtils.extractErrorMessage)(e);
  }
  return {
    resp,
    errorMessage
  };
}
async function retryIndexExistsCheck(destIndex) {
  let retryCount = 15;
  let resp = await checkIndexExists(destIndex);
  let indexExists = resp.resp && resp.resp[destIndex] && resp.resp[destIndex].exists;
  while (retryCount > 1 && !indexExists) {
    retryCount--;
    await delay(1000);
    resp = await checkIndexExists(destIndex);
    indexExists = resp.resp && resp.resp[destIndex] && resp.resp[destIndex].exists;
  }
  if (indexExists) {
    return {
      success: true,
      indexExists: true
    };
  }
  return {
    success: false,
    indexExists: false,
    ...(resp.errorMessage !== undefined ? {
      errorMessage: resp.errorMessage
    } : {})
  };
}
const createKibanaDataView = async (destinationIndex, dataViewsService, timeFieldName, callback) => {
  const response = {
    success: false,
    message: ''
  };
  const dataViewName = destinationIndex;
  const exists = await retryIndexExistsCheck(destinationIndex);
  if ((exists === null || exists === void 0 ? void 0 : exists.success) === true) {
    // index exists - create data view
    if ((exists === null || exists === void 0 ? void 0 : exists.indexExists) === true) {
      try {
        const dataView = await dataViewsService.createAndSave({
          title: dataViewName,
          ...(timeFieldName ? {
            timeFieldName
          } : {})
        }, false, true);
        response.success = true;
        response.message = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.createDataViewSuccessMessage', {
          defaultMessage: 'Kibana data view {dataViewName} created.',
          values: {
            dataViewName
          }
        });
        response.dataViewId = dataView.id;
      } catch (e) {
        // handle data view creation error
        if (e instanceof _public.DuplicateDataViewError) {
          response.error = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.duplicateDataViewErrorMessageError', {
            defaultMessage: 'The data view {dataViewName} already exists.',
            values: {
              dataViewName
            }
          });
          response.message = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.duplicateDataViewErrorMessage', {
            defaultMessage: 'An error occurred creating the Kibana data view:'
          });
        } else {
          response.error = (0, _mlErrorUtils.extractErrorMessage)(e);
          response.message = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.createDataViewErrorMessage', {
            defaultMessage: 'An error occurred creating the Kibana data view:'
          });
        }
      }
    }
  } else {
    // Ran out of retries or there was a problem checking index exists
    if (exists !== null && exists !== void 0 && exists.errorMessage) {
      response.error = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorCheckingDestinationIndexDataFrameAnalyticsJob', {
        defaultMessage: '{errorMessage}',
        values: {
          errorMessage: exists.errorMessage
        }
      });
      response.message = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorOccurredCheckingDestinationIndexDataFrameAnalyticsJob', {
        defaultMessage: 'An error occurred checking destination index exists.'
      });
    } else {
      response.error = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.destinationIndexNotCreatedForDataFrameAnalyticsJob', {
        defaultMessage: 'Destination index has not yet been created.'
      });
      response.message = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.unableToCreateDataViewForDataFrameAnalyticsJob', {
        defaultMessage: 'Unable to create data view.'
      });
    }
  }
  if (callback !== undefined) {
    callback({
      error: response.error,
      message: response.message
    });
  }
  return response;
};
exports.createKibanaDataView = createKibanaDataView;