"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JobsExportService = void 0;
var _filesaver = require("@elastic/filesaver");
var _calendars = require("../../../../../common/constants/calendars");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// @ts-expect-error

class JobsExportService {
  constructor(_mlApiServices) {
    this._mlApiServices = _mlApiServices;
  }
  async exportAnomalyDetectionJobs(jobIds) {
    const configs = await Promise.all(jobIds.map(id => this._mlApiServices.jobs.jobForCloning(id, true)));
    this._export(configs, 'anomaly-detector');
  }
  async exportDataframeAnalyticsJobs(jobIds) {
    const {
      data_frame_analytics: configs
    } = await this._mlApiServices.dataFrameAnalytics.getDataFrameAnalytics(jobIds.join(','), true);
    this._export(configs, 'data-frame-analytics');
  }
  _export(configs, jobType) {
    const configsForExport = configs.length === 1 ? configs[0] : configs;
    const blob = new Blob([JSON.stringify(configsForExport, null, 2)], {
      type: 'application/json'
    });
    const fileName = this._createFileName(jobType);
    (0, _filesaver.saveAs)(blob, fileName);
  }
  _createFileName(jobType) {
    return (jobType === 'anomaly-detector' ? 'anomaly_detection' : 'data_frame_analytics') + '_jobs.json';
  }
  async getJobDependencies(jobs) {
    const calendars = await this._mlApiServices.calendars();

    // create a map of all jobs in groups
    const groups = jobs.reduce((acc, cur) => {
      if (Array.isArray(cur.groups)) {
        cur.groups.forEach(g => {
          if (acc[g] === undefined) {
            acc[g] = [];
          }
          acc[g].push(cur.job_id);
        });
      }
      return acc;
    }, {});
    const isGroup = id => groups[id] !== undefined;

    // create a map of all calendars in jobs
    const calendarsPerJob = calendars.reduce((acc, cur) => {
      cur.job_ids.forEach(jId => {
        if (jId === _calendars.GLOBAL_CALENDAR) {
          // add the calendar to all jobs
          jobs.forEach(j => {
            if (acc[j.job_id] === undefined) {
              acc[j.job_id] = [];
            }
            acc[j.job_id].push(cur.calendar_id);
          });
        } else if (isGroup(jId)) {
          // add the calendar to every job in this group
          groups[jId].forEach(jId2 => {
            if (acc[jId2] === undefined) {
              acc[jId2] = [];
            }
            acc[jId2].push(cur.calendar_id);
          });
        } else {
          // add the calendar to just this job
          if (acc[jId] === undefined) {
            acc[jId] = [];
          }
          acc[jId].push(cur.calendar_id);
        }
      });
      return acc;
    }, {});

    // create a map of all filters in jobs,
    // by extracting the filters from the job's detectors
    const filtersPerJob = jobs.reduce((acc, cur) => {
      if (acc[cur.job_id] === undefined) {
        acc[cur.job_id] = [];
      }
      cur.analysis_config.detectors.forEach(d => {
        if (d.custom_rules !== undefined) {
          d.custom_rules.forEach(r => {
            if (r.scope !== undefined) {
              Object.values(r.scope).forEach(scope => {
                acc[cur.job_id].push(scope.filter_id);
              });
            }
          });
        }
      });
      return acc;
    }, {});
    return jobs.map(j => {
      var _ref, _ref2;
      const jobId = j.job_id;
      return {
        jobId,
        calendarIds: (_ref = [...new Set(calendarsPerJob[jobId])]) !== null && _ref !== void 0 ? _ref : [],
        filterIds: (_ref2 = [...new Set(filtersPerJob[jobId])]) !== null && _ref2 !== void 0 ? _ref2 : []
      };
    });
  }
}
exports.JobsExportService = JobsExportService;