"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ESQL_GEO_SHAPE_TYPE = exports.ESQL_GEO_POINT_TYPE = void 0;
exports.getESQLMeta = getESQLMeta;
exports.getFieldType = getFieldType;
exports.getFields = getFields;
exports.isGeometryColumn = isGeometryColumn;
exports.verifyGeometryColumn = verifyGeometryColumn;
var _i18n = require("@kbn/i18n");
var _esqlUtils = require("@kbn/esql-utils");
var _constants = require("../../../../common/constants");
var _kibana_services = require("../../../kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// ESQL_GEO_POINT_TYPE !== ES_GEO_FIELD_TYPE.GEO_POINT
// ES_GEO_FIELD_TYPE.GEO_POINT is a field type from an Elasticsearch index mapping
// ESQL_GEO_POINT_TYPE is a column type from an ESQL response
const ESQL_GEO_POINT_TYPE = exports.ESQL_GEO_POINT_TYPE = 'geo_point';

// ESQL_GEO_SHAPE_TYPE !== ES_GEO_FIELD_TYPE.GEO_SHAPE
// ES_GEO_FIELD_TYPE.GEO_SHAPE is a field type from an Elasticsearch index mapping
// ESQL_GEO_SHAPE_TYPE is a column type from an ESQL response
const ESQL_GEO_SHAPE_TYPE = exports.ESQL_GEO_SHAPE_TYPE = 'geo_shape';
function isGeometryColumn(column) {
  return [ESQL_GEO_POINT_TYPE, ESQL_GEO_SHAPE_TYPE].includes(column.type);
}
function verifyGeometryColumn(columns) {
  const geometryColumns = columns.filter(isGeometryColumn);
  if (geometryColumns.length === 0) {
    throw new Error(_i18n.i18n.translate('xpack.maps.source.esql.noGeometryColumnErrorMsg', {
      defaultMessage: 'Elasticsearch ES|QL query does not have a geometry column.'
    }));
  }
  if (geometryColumns.length > 1) {
    throw new Error(_i18n.i18n.translate('xpack.maps.source.esql.multipleGeometryColumnErrorMsg', {
      defaultMessage: `Elasticsearch ES|QL query has {count} geometry columns when only 1 is allowed. Use 'DROP' or 'KEEP' to narrow columns.`,
      values: {
        count: geometryColumns.length
      }
    }));
  }
}
async function getESQLMeta(esql) {
  const adhocDataView = await (0, _esqlUtils.getESQLAdHocDataview)((0, _esqlUtils.getIndexPatternFromESQLQuery)(esql), (0, _kibana_services.getIndexPatternService)());
  return {
    columns: await (0, _esqlUtils.getESQLQueryColumnsRaw)({
      esqlQuery: esql,
      search: (0, _kibana_services.getData)().search.search
    }),
    adhocDataViewId: adhocDataView.id,
    ...getFields(adhocDataView)
  };
}

/*
 * Map column.type to field type
 * Supported column types https://www.elastic.co/guide/en/elasticsearch/reference/master/esql-limitations.html#_supported_types
 */
function getFieldType(column) {
  switch (column.type) {
    case 'boolean':
    case 'date':
    case 'ip':
    case 'keyword':
    case 'text':
      return 'string';
    case 'double':
    case 'int':
    case 'long':
    case 'unsigned_long':
      return 'number';
    default:
      return undefined;
  }
}
function getFields(dataView) {
  const dateFields = [];
  const geoFields = [];
  dataView.fields.forEach(field => {
    if (field.type === 'date') {
      dateFields.push(field.name);
    } else if (field.type === _constants.ES_GEO_FIELD_TYPE.GEO_POINT || field.type === _constants.ES_GEO_FIELD_TYPE.GEO_SHAPE) {
      geoFields.push(field.name);
    }
  });
  return {
    dateFields,
    geoFields
  };
}