"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGetLogsRatesService = createGetLogsRatesService;
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createGetLogsRatesService(params) {
  return async ({
    esClient,
    identifyingMetadata,
    serviceNames,
    timeFrom,
    timeTo
  }) => {
    const esResponse = await esClient.search({
      index: 'logs-*-*',
      size: 0,
      query: {
        bool: {
          filter: [{
            exists: {
              // For now, we don't want to count APM server logs or any other logs that don't have the log.level field.
              field: 'log.level'
            }
          }, {
            terms: {
              [identifyingMetadata]: serviceNames
            }
          }, {
            range: {
              ['@timestamp']: {
                gte: timeFrom,
                lte: timeTo,
                format: 'epoch_millis'
              }
            }
          }]
        }
      },
      aggs: {
        services: {
          terms: {
            field: identifyingMetadata
          },
          aggs: {
            logErrors: {
              terms: {
                field: 'log.level',
                include: ['error', 'ERROR']
              }
            }
          }
        }
      }
    });
    const aggregations = esResponse.aggregations;
    const buckets = aggregations === null || aggregations === void 0 ? void 0 : aggregations.services.buckets;
    return buckets ? buckets.reduce((acc, bucket) => {
      var _logErrorBuckets$;
      const logCount = bucket.doc_count;
      const logErrorBuckets = bucket.logErrors.buckets;
      const logErrorCount = (_logErrorBuckets$ = logErrorBuckets[0]) === null || _logErrorBuckets$ === void 0 ? void 0 : _logErrorBuckets$.doc_count;
      return {
        ...acc,
        [bucket.key]: {
          logRatePerMinute: (0, _utils.getLogRatePerMinute)({
            logCount,
            timeFrom,
            timeTo
          }),
          logErrorRate: logErrorCount ? (0, _utils.getLogErrorRate)({
            logCount,
            logErrorCount
          }) : null
        }
      };
    }, {}) : {};
  };
}