"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeCreateAction = executeCreateAction;
exports.getVisualizationMap = exports.getDatasourceMap = void 0;
exports.isCreateActionCompatible = isCreateActionCompatible;
exports.setVisualizationMap = exports.setDatasourceMap = void 0;
var _common = require("@kbn/kibana-utils-plugin/common");
var _visualizationUtils = require("@kbn/visualization-utils");
var _public = require("@kbn/ui-actions-plugin/public");
var _esqlUtils = require("@kbn/esql-utils");
var _lens_suggestions_api = require("../../lens_suggestions_api");
var _id_generator = require("../../id_generator");
var _edit_action_helpers = require("./edit_action_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// datasourceMap and visualizationMap setters/getters
const [getVisualizationMap, setVisualizationMap] = (0, _common.createGetterSetter)('VisualizationMap', false);
exports.setVisualizationMap = setVisualizationMap;
exports.getVisualizationMap = getVisualizationMap;
const [getDatasourceMap, setDatasourceMap] = (0, _common.createGetterSetter)('DatasourceMap', false);
exports.setDatasourceMap = setDatasourceMap;
exports.getDatasourceMap = getDatasourceMap;
function isCreateActionCompatible(core) {
  return core.uiSettings.get(_esqlUtils.ENABLE_ESQL);
}
async function executeCreateAction({
  deps,
  core,
  api
}) {
  var _suggestionsApi;
  const isCompatibleAction = isCreateActionCompatible(core);
  const getFallbackDataView = async () => {
    const indexName = await (0, _esqlUtils.getIndexForESQLQuery)({
      dataViews: deps.dataViews
    });
    if (!indexName) return null;
    const dataView = await (0, _esqlUtils.getESQLAdHocDataview)(indexName, deps.dataViews);
    return dataView;
  };
  const dataView = await getFallbackDataView();
  if (!isCompatibleAction || !dataView) {
    throw new _public.IncompatibleActionError();
  }
  const visualizationMap = getVisualizationMap();
  const datasourceMap = getDatasourceMap();
  const defaultIndex = dataView.getIndexPattern();
  const defaultEsqlQuery = {
    esql: `FROM ${defaultIndex} | LIMIT 10`
  };

  // For the suggestions api we need only the columns
  // so we are requesting them with limit 0
  // this is much more performant than requesting
  // all the table
  const abortController = new AbortController();
  const columns = await (0, _esqlUtils.getESQLQueryColumns)({
    esqlQuery: `from ${defaultIndex}`,
    search: deps.data.search.search,
    signal: abortController.signal
  });
  const context = {
    dataViewSpec: dataView.toSpec(false),
    fieldName: '',
    textBasedColumns: columns,
    query: defaultEsqlQuery
  };

  // get the initial attributes from the suggestions api
  const allSuggestions = (_suggestionsApi = (0, _lens_suggestions_api.suggestionsApi)({
    context,
    dataView,
    datasourceMap,
    visualizationMap
  })) !== null && _suggestionsApi !== void 0 ? _suggestionsApi : [];

  // Lens might not return suggestions for some cases, i.e. in case of errors
  if (!allSuggestions.length) return undefined;
  const [firstSuggestion] = allSuggestions;
  const attrs = (0, _visualizationUtils.getLensAttributesFromSuggestion)({
    filters: [],
    query: defaultEsqlQuery,
    suggestion: firstSuggestion,
    dataView
  });
  const embeddable = await api.addNewPanel({
    panelType: 'lens',
    initialState: {
      attributes: attrs,
      id: (0, _id_generator.generateId)()
    }
  });
  // open the flyout if embeddable has been created successfully
  if (embeddable) {
    const deletePanel = () => {
      api.removePanel(embeddable.id);
    };
    (0, _edit_action_helpers.executeEditAction)({
      embeddable,
      startDependencies: deps,
      isNewPanel: true,
      deletePanel,
      ...core
    });
  }
}