"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.adjustColumnReferences = adjustColumnReferences;
exports.adjustColumnReferencesForChangedColumn = adjustColumnReferencesForChangedColumn;
exports.canTransition = canTransition;
exports.copyColumn = copyColumn;
exports.deleteColumn = deleteColumn;
exports.getColumnOrder = getColumnOrder;
exports.getErrorMessages = getErrorMessages;
exports.getExistingColumnGroups = getExistingColumnGroups;
exports.getManagedColumnsFrom = getManagedColumnsFrom;
exports.getMetricOperationTypes = getMetricOperationTypes;
exports.getReferenceRoot = getReferenceRoot;
exports.getReferencedColumnIds = getReferencedColumnIds;
exports.hasTermsWithManyBuckets = hasTermsWithManyBuckets;
exports.insertNewColumn = insertNewColumn;
exports.insertOrReplaceColumn = insertOrReplaceColumn;
exports.isColumnTransferable = isColumnTransferable;
exports.isColumnValidAsReference = isColumnValidAsReference;
exports.isOperationAllowedAsReference = isOperationAllowedAsReference;
exports.isReferenced = isReferenced;
exports.reorderByGroups = reorderByGroups;
exports.replaceColumn = replaceColumn;
exports.resetIncomplete = resetIncomplete;
exports.updateColumnLabel = updateColumnLabel;
exports.updateColumnParam = updateColumnParam;
exports.updateDefaultLabels = updateDefaultLabels;
exports.updateLayerIndexPattern = updateLayerIndexPattern;
var _lodash = require("lodash");
var _memoizeOne = _interopRequireDefault(require("memoize-one"));
var _public = require("@kbn/data-plugin/public");
var _definitions = require("./definitions");
var _operations = require("./operations");
var _id_generator = require("../../../id_generator");
var _formula = require("./definitions/formula");
var _document_field = require("../document_field");
var _helpers = require("./definitions/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function copyColumn({
  layers,
  source,
  target
}) {
  return createCopiedColumn(layers, target, source);
}
function createCopiedColumn(layers, target, source) {
  const sourceLayer = layers[source.layerId];
  const sourceColumn = sourceLayer.columns[source.columnId];
  const targetLayer = layers[target.layerId];
  let columns = {
    ...targetLayer.columns
  };
  if ('references' in sourceColumn) {
    const def = _definitions.operationDefinitionMap[sourceColumn.operationType];
    if ('createCopy' in def) {
      return def.createCopy(layers, source, target, _definitions.operationDefinitionMap); // Allow managed references to recursively insert new columns
    }
    const referenceColumns = sourceColumn.references.reduce((refs, sourceRef) => {
      const newRefId = (0, _id_generator.generateId)();
      return {
        ...refs,
        [newRefId]: {
          ...sourceLayer.columns[sourceRef]
        }
      };
    }, {});
    columns = {
      ...columns,
      ...referenceColumns,
      [target.columnId]: {
        ...sourceColumn,
        references: Object.keys(referenceColumns)
      }
    };
  } else {
    columns = {
      ...columns,
      [target.columnId]: {
        ...sourceColumn
      }
    };
  }
  return {
    ...layers,
    [target.layerId]: adjustColumnReferences({
      ...targetLayer,
      columns,
      columnOrder: getColumnOrder({
        ...targetLayer,
        columns
      })
    })
  };
}
function insertOrReplaceColumn(args) {
  if (args.layer.columns[args.columnId]) {
    return replaceColumn(args);
  }
  return insertNewColumn(args);
}
function ensureCompatibleParamsAreMoved(column, referencedOperation, previousColumn) {
  const newColumn = {
    ...column
  };
  if (referencedOperation.filterable) {
    newColumn.filter = previousColumn.filter;
  }
  if (referencedOperation.shiftable) {
    newColumn.timeShift = previousColumn.timeShift;
  }
  if (referencedOperation.timeScalingMode !== 'disabled') {
    newColumn.timeScale = previousColumn.timeScale;
  }
  return newColumn;
}
const insertReferences = ({
  layer,
  references,
  requiredReferences,
  indexPattern,
  visualizationGroups,
  targetGroup
}) => {
  references.forEach(reference => {
    const validOperations = requiredReferences.filter(validation => isOperationAllowedAsReference({
      validation,
      operationType: reference.op,
      indexPattern
    }));
    if (!validOperations.length) {
      throw new Error(`Can't create reference, ${reference.op} has a validation function which doesn't allow any operations`);
    }
  });
  const referenceIds = [];
  references.forEach(reference => {
    const operation = _definitions.operationDefinitionMap[reference.op];
    if (operation.input === 'none') {
      layer = insertNewColumn({
        layer,
        columnId: reference.columnId,
        op: operation.type,
        indexPattern,
        columnParams: {
          ...reference.columnParams
        },
        incompleteParams: reference.incompleteParams,
        initialParams: reference.initialParams,
        ...(reference.references ? {
          references: reference.references
        } : {}),
        visualizationGroups,
        targetGroup
      });
      referenceIds.push(reference.columnId);
      return;
    }
    const field = operation.input === 'field' && reference.field && operation.getPossibleOperationForField(reference.field) ? reference.field : undefined;
    if (field) {
      // Recursively update the layer for each new reference
      layer = insertNewColumn({
        layer,
        columnId: reference.columnId,
        op: operation.type,
        indexPattern,
        field,
        incompleteParams: reference.incompleteParams,
        initialParams: reference.initialParams,
        columnParams: {
          ...reference.columnParams
        },
        visualizationGroups,
        targetGroup
      });
      referenceIds.push(reference.columnId);
      return;
    }
  });
  return {
    layer,
    referenceIds
  };
};
const generateNewReferences = ({
  op,
  incompleteFieldOperation,
  incompleteFieldName,
  columnParams,
  layer,
  requiredReferences,
  indexPattern,
  visualizationGroups,
  targetGroup
}) => {
  const referenceIds = requiredReferences.map(validation => {
    const validOperations = Object.values(_definitions.operationDefinitionMap).filter(({
      type
    }) => isOperationAllowedAsReference({
      validation,
      operationType: type,
      indexPattern
    }));
    if (!validOperations.length) {
      throw new Error(`Can't create reference, ${op} has a validation function which doesn't allow any operations`);
    }
    const newId = (0, _id_generator.generateId)();
    if (incompleteFieldOperation && incompleteFieldName) {
      var _validFields$;
      const validFields = indexPattern.fields.filter(validField => validField.name === incompleteFieldName);
      layer = insertNewColumn({
        layer,
        columnId: newId,
        op: incompleteFieldOperation,
        indexPattern,
        field: (_validFields$ = validFields[0]) !== null && _validFields$ !== void 0 ? _validFields$ : _document_field.documentField,
        visualizationGroups,
        columnParams,
        targetGroup
      });
    }
    if (validOperations.length === 1) {
      const def = validOperations[0];
      let validFields = def.input === 'field' ? indexPattern.fields.filter(def.getPossibleOperationForField) : [];
      if (incompleteFieldName) {
        validFields = validFields.filter(validField => validField.name === incompleteFieldName);
      }
      if (def.input === 'none') {
        layer = insertNewColumn({
          layer,
          columnId: newId,
          op: def.type,
          indexPattern,
          visualizationGroups,
          targetGroup
        });
      } else if (validFields.length === 1) {
        // Recursively update the layer for each new reference
        layer = insertNewColumn({
          layer,
          columnId: newId,
          op: def.type,
          indexPattern,
          field: validFields[0],
          visualizationGroups,
          targetGroup
        });
      } else {
        layer = {
          ...layer,
          incompleteColumns: {
            ...layer.incompleteColumns,
            [newId]: {
              operationType: def.type
            }
          }
        };
      }
    }
    return newId;
  });
  return {
    layer,
    referenceIds
  };
};

// Insert a column into an empty ID. The field parameter is required when constructing
// a field-based operation, but will cause the function to fail for any other type of operation.
function insertNewColumn({
  op,
  layer,
  columnId,
  field,
  indexPattern,
  visualizationGroups,
  targetGroup,
  incompleteParams,
  incompleteFieldName,
  incompleteFieldOperation,
  columnParams,
  initialParams,
  references,
  respectOrder
}) {
  var _columnId, _layer$incompleteColu;
  const operationDefinition = _definitions.operationDefinitionMap[op];
  if (!operationDefinition) {
    throw new Error('No suitable operation found for given parameters');
  }
  if (layer.columns[columnId]) {
    throw new Error(`Can't insert a column with an ID that is already in use`);
  }
  const baseOptions = {
    indexPattern,
    previousColumn: {
      ...incompleteParams,
      ...initialParams,
      ...layer.columns[columnId]
    }
  };
  if (operationDefinition.input === 'none' || operationDefinition.input === 'managedReference') {
    if (field) {
      throw new Error(`Can't create operation ${op} with the provided field ${field.name}`);
    }
    if (operationDefinition.input === 'managedReference') {
      // TODO: need to create on the fly the new columns for Formula,
      // like we do for fullReferences to show a seamless transition
    }
    const possibleOperation = operationDefinition.getPossibleOperation(indexPattern);
    const isBucketed = Boolean(possibleOperation === null || possibleOperation === void 0 ? void 0 : possibleOperation.isBucketed);
    const addOperationFn = isBucketed ? addBucket : addMetric;
    const buildColumnFn = columnParams ? operationDefinition.buildColumn({
      ...baseOptions,
      layer
    }, columnParams) : operationDefinition.buildColumn({
      ...baseOptions,
      layer
    });
    return updateDefaultLabels(addOperationFn(layer, buildColumnFn, columnId, visualizationGroups, targetGroup, respectOrder), indexPattern);
  }
  if (operationDefinition.input === 'fullReference') {
    if (field) {
      throw new Error(`Reference-based operations can't take a field as input when creating`);
    }
    let tempLayer = {
      ...layer
    };
    let referenceIds = [];
    if (references) {
      const result = insertReferences({
        layer: tempLayer,
        references,
        requiredReferences: operationDefinition.requiredReferences,
        indexPattern,
        visualizationGroups,
        targetGroup
      });
      [tempLayer, referenceIds] = [result.layer, result.referenceIds];
    } else {
      const result = generateNewReferences({
        op,
        incompleteFieldName,
        incompleteFieldOperation,
        columnParams,
        layer: tempLayer,
        requiredReferences: operationDefinition.requiredReferences,
        indexPattern,
        visualizationGroups,
        targetGroup
      });
      [tempLayer, referenceIds] = [result.layer, result.referenceIds];
    }
    const possibleOperation = operationDefinition.getPossibleOperation(indexPattern);
    if (!possibleOperation) {
      throw new Error(`Can't create operation ${op} because it's incompatible with the data view`);
    }
    const isBucketed = Boolean(possibleOperation.isBucketed);
    const addOperationFn = isBucketed ? addBucket : addMetric;
    const buildColumnFn = columnParams ? operationDefinition.buildColumn({
      ...baseOptions,
      layer: tempLayer,
      referenceIds
    }, columnParams) : operationDefinition.buildColumn({
      ...baseOptions,
      layer: tempLayer,
      referenceIds
    });
    return updateDefaultLabels(addOperationFn(tempLayer, buildColumnFn, columnId, visualizationGroups, targetGroup, respectOrder), indexPattern);
  }
  const invalidFieldName = (_columnId = ((_layer$incompleteColu = layer.incompleteColumns) !== null && _layer$incompleteColu !== void 0 ? _layer$incompleteColu : {})[columnId]) === null || _columnId === void 0 ? void 0 : _columnId.sourceField;
  const invalidField = invalidFieldName ? indexPattern.getFieldByName(invalidFieldName) : undefined;
  if (!field && invalidField) {
    const possibleOperation = operationDefinition.getPossibleOperationForField(invalidField);
    if (!possibleOperation) {
      throw new Error(`Tried to create an invalid operation ${operationDefinition.type} using previously selected field ${invalidField.name}`);
    }
    const isBucketed = Boolean(possibleOperation.isBucketed);
    if (isBucketed) {
      return updateDefaultLabels(addBucket(layer, operationDefinition.buildColumn({
        ...baseOptions,
        layer,
        field: invalidField
      }), columnId, visualizationGroups, targetGroup, respectOrder), indexPattern);
    } else {
      return updateDefaultLabels(addMetric(layer, operationDefinition.buildColumn({
        ...baseOptions,
        layer,
        field: invalidField
      }), columnId), indexPattern);
    }
  } else if (!field) {
    var _layer$incompleteColu2;
    // Labels don't need to be updated because it's incomplete
    return {
      ...layer,
      incompleteColumns: {
        ...((_layer$incompleteColu2 = layer.incompleteColumns) !== null && _layer$incompleteColu2 !== void 0 ? _layer$incompleteColu2 : {}),
        [columnId]: {
          operationType: op
        }
      }
    };
  }
  const possibleOperation = operationDefinition.getPossibleOperationForField(field);
  if (!possibleOperation) {
    var _layer$incompleteColu3;
    return {
      ...layer,
      incompleteColumns: {
        ...((_layer$incompleteColu3 = layer.incompleteColumns) !== null && _layer$incompleteColu3 !== void 0 ? _layer$incompleteColu3 : {}),
        [columnId]: {
          operationType: op,
          sourceField: field.name
        }
      }
    };
  }
  const newColumn = operationDefinition.buildColumn({
    ...baseOptions,
    layer,
    field
  }, columnParams);
  const isBucketed = Boolean(possibleOperation.isBucketed);
  const addOperationFn = isBucketed ? addBucket : addMetric;
  return updateDefaultLabels(addOperationFn(layer, newColumn, columnId, visualizationGroups, targetGroup, respectOrder), indexPattern);
}
function replaceFormulaColumn({
  operationDefinition,
  layer,
  previousColumn,
  indexPattern,
  previousDefinition,
  columnId
}, {
  shouldResetLabel
}) {
  const baseOptions = {
    columns: layer.columns,
    previousColumn,
    indexPattern
  };
  let tempLayer = layer;
  const newColumn = operationDefinition.buildColumn({
    ...baseOptions,
    layer: tempLayer
  }, 'params' in previousColumn ? previousColumn.params : undefined, _definitions.operationDefinitionMap);

  // now remove the previous references
  if (previousDefinition.input === 'fullReference') {
    previousColumn.references.forEach(id => {
      tempLayer = deleteColumn({
        layer: tempLayer,
        columnId: id,
        indexPattern
      });
    });
  }
  const basicLayer = {
    ...tempLayer,
    columns: {
      ...tempLayer.columns,
      [columnId]: newColumn
    }
  };
  // rebuild the references again for the specific AST generated
  let newLayer;
  try {
    newLayer = newColumn.params.formula ? (0, _formula.insertOrReplaceFormulaColumn)(columnId, newColumn, basicLayer, {
      indexPattern
    }).layer : basicLayer;
  } catch (e) {
    newLayer = basicLayer;
  }

  // when coming to Formula keep the custom label
  const regeneratedColumn = newLayer.columns[columnId];
  if (!shouldResetLabel && previousColumn.customLabel) {
    regeneratedColumn.customLabel = true;
    regeneratedColumn.label = previousColumn.label;
  }
  return updateDefaultLabels(adjustColumnReferencesForChangedColumn({
    ...tempLayer,
    columnOrder: getColumnOrder(newLayer),
    columns: newLayer.columns
  }, columnId), indexPattern);
}
function replaceColumn({
  layer,
  columnId,
  indexPattern,
  op,
  field,
  visualizationGroups,
  initialParams,
  shouldResetLabel,
  shouldCombineField
}) {
  const previousColumn = layer.columns[columnId];
  if (!previousColumn) {
    throw new Error(`Can't replace column because there is no prior column`);
  }
  const isNewOperation = op !== previousColumn.operationType;
  const operationDefinition = _definitions.operationDefinitionMap[op];
  const previousDefinition = _definitions.operationDefinitionMap[previousColumn.operationType];
  if (!operationDefinition) {
    throw new Error('No suitable operation found for given parameters');
  }
  const baseOptions = {
    columns: layer.columns,
    indexPattern,
    previousColumn
  };
  if (isNewOperation) {
    let tempLayer = {
      ...layer
    };
    tempLayer = resetIncomplete(tempLayer, columnId);
    if (previousDefinition.input === 'managedReference' && operationDefinition.input !== previousDefinition.input) {
      // If the transition is incomplete, leave the managed state until it's finished.
      tempLayer = removeOrphanedColumns(previousDefinition, previousColumn, tempLayer, indexPattern);
      const hypotheticalLayer = insertNewColumn({
        layer: tempLayer,
        columnId,
        indexPattern,
        op,
        field,
        visualizationGroups,
        incompleteParams: previousColumn
      });

      // if the formula label is not the default one, propagate it to the new operation
      if (!shouldResetLabel && previousColumn.customLabel && hypotheticalLayer.columns[columnId] && previousColumn.label !== previousDefinition.getDefaultLabel(previousColumn, tempLayer.columns, indexPattern)) {
        hypotheticalLayer.columns[columnId].customLabel = true;
        hypotheticalLayer.columns[columnId].label = previousColumn.label;
      }
      if (hypotheticalLayer.incompleteColumns && hypotheticalLayer.incompleteColumns[columnId]) {
        return {
          ...layer,
          incompleteColumns: hypotheticalLayer.incompleteColumns
        };
      } else {
        return hypotheticalLayer;
      }
    }
    if (operationDefinition.input === 'fullReference') {
      return applyReferenceTransition({
        layer: tempLayer,
        columnId,
        previousColumn,
        op,
        indexPattern,
        visualizationGroups
      });
    }

    // Makes common inferences about what the user meant when switching away from a reference:
    // 1. Switching from "Differences of max" to "max" will promote as-is
    // 2. Switching from "Differences of avg of bytes" to "max" will keep the field, but change operation
    if (previousDefinition.input === 'fullReference' && previousColumn.references.length === 1) {
      const previousReferenceId = previousColumn.references[0];
      const referenceColumn = layer.columns[previousReferenceId];
      if (referenceColumn) {
        const referencedOperation = _definitions.operationDefinitionMap[referenceColumn.operationType];
        if (referencedOperation.type === op) {
          // Unit tests are labelled as case a1, case a2
          tempLayer = deleteColumn({
            layer: tempLayer,
            columnId: previousReferenceId,
            indexPattern
          });

          // do not forget to move over also any filter/shift/anything (if compatible)
          // from the reference definition to the new operation.
          const column = ensureCompatibleParamsAreMoved(copyCustomLabel({
            ...referenceColumn
          }, previousColumn), referencedOperation, previousColumn);
          tempLayer = {
            ...tempLayer,
            columnOrder: getColumnOrder(tempLayer),
            columns: {
              ...tempLayer.columns,
              [columnId]: column
            }
          };
          return updateDefaultLabels(adjustColumnReferencesForChangedColumn(tempLayer, columnId), indexPattern);
        } else if (!field && 'sourceField' in referenceColumn && referencedOperation.input === 'field' && operationDefinition.input === 'field') {
          // Unit test is case a3
          const matchedField = indexPattern.getFieldByName(referenceColumn.sourceField);
          if (matchedField && operationDefinition.getPossibleOperationForField(matchedField)) {
            field = matchedField;
          }
        }
      }
    }

    // TODO: Refactor all this to be more generic and know less about Formula
    // if managed it has to look at the full picture to have a seamless transition
    if (operationDefinition.input === 'managedReference') {
      return replaceFormulaColumn({
        operationDefinition,
        layer: tempLayer,
        previousColumn,
        indexPattern,
        previousDefinition,
        columnId
      }, {
        shouldResetLabel
      });
    }
    if (operationDefinition.input === 'none') {
      let newColumn = operationDefinition.buildColumn({
        ...baseOptions,
        layer: tempLayer
      });
      newColumn = copyCustomLabel(newColumn, previousColumn);
      tempLayer = removeOrphanedColumns(previousDefinition, previousColumn, tempLayer, indexPattern);
      const newLayer = {
        ...tempLayer,
        columns: {
          ...tempLayer.columns,
          [columnId]: newColumn
        }
      };
      return updateDefaultLabels(adjustColumnReferencesForChangedColumn({
        ...newLayer,
        columnOrder: getColumnOrder(newLayer)
      }, columnId), indexPattern);
    }
    if (!field) {
      var _tempLayer$incomplete;
      let incompleteColumn = {
        operationType: op
      };
      // if no field is available perform a full clean of the column from the layer
      if (previousDefinition.input === 'fullReference') {
        const previousReferenceId = previousColumn.references[0];
        const referenceColumn = layer.columns[previousReferenceId];
        if (referenceColumn) {
          const referencedOperation = _definitions.operationDefinitionMap[referenceColumn.operationType];
          incompleteColumn = ensureCompatibleParamsAreMoved(incompleteColumn, referencedOperation, previousColumn);
        }
      }
      return {
        ...tempLayer,
        incompleteColumns: {
          ...((_tempLayer$incomplete = tempLayer.incompleteColumns) !== null && _tempLayer$incomplete !== void 0 ? _tempLayer$incomplete : {}),
          [columnId]: incompleteColumn
        }
      };
    }
    const validOperation = operationDefinition.getPossibleOperationForField(field);
    if (!validOperation) {
      var _tempLayer$incomplete2;
      return {
        ...tempLayer,
        incompleteColumns: {
          ...((_tempLayer$incomplete2 = tempLayer.incompleteColumns) !== null && _tempLayer$incomplete2 !== void 0 ? _tempLayer$incomplete2 : {}),
          [columnId]: {
            operationType: op
          }
        }
      };
    }
    tempLayer = removeOrphanedColumns(previousDefinition, previousColumn, tempLayer, indexPattern);
    let newColumn = operationDefinition.buildColumn({
      ...baseOptions,
      layer: tempLayer,
      field
    });
    if (!shouldResetLabel) {
      newColumn = copyCustomLabel(newColumn, previousColumn);
    }
    const newLayer = {
      ...tempLayer,
      columns: {
        ...tempLayer.columns,
        [columnId]: newColumn
      }
    };
    return updateDefaultLabels(adjustColumnReferencesForChangedColumn({
      ...newLayer,
      columnOrder: getColumnOrder(newLayer)
    }, columnId), indexPattern);
  } else if (operationDefinition.input === 'field' && field && 'sourceField' in previousColumn && (previousColumn.sourceField !== field.name || operationDefinition !== null && operationDefinition !== void 0 && operationDefinition.getParamsForMultipleFields)) {
    // Same operation, new field
    let newColumn = operationDefinition.onFieldChange(previousColumn, field, shouldCombineField ? initialParams === null || initialParams === void 0 ? void 0 : initialParams.params : undefined);
    if (!shouldResetLabel) {
      newColumn = copyCustomLabel(newColumn, previousColumn);
    }
    const newLayer = resetIncomplete({
      ...layer,
      columns: {
        ...layer.columns,
        [columnId]: newColumn
      }
    }, columnId);
    return updateDefaultLabels(adjustColumnReferencesForChangedColumn({
      ...newLayer,
      columnOrder: getColumnOrder(newLayer)
    }, columnId), indexPattern);
  } else if (operationDefinition.input === 'managedReference') {
    // Just changing a param in a formula column should trigger
    // a full formula regeneration for side effects on referenced columns
    return replaceFormulaColumn({
      operationDefinition,
      layer,
      previousColumn,
      indexPattern,
      previousDefinition,
      columnId
    }, {
      shouldResetLabel
    });
  } else {
    throw new Error('nothing changed');
  }
}
function removeOrphanedColumns(previousDefinition, previousColumn, tempLayer, indexPattern) {
  let newLayer = tempLayer;
  if (previousDefinition.input === 'managedReference') {
    const [columnId] = Object.entries(tempLayer.columns).find(([_, currColumn]) => currColumn === previousColumn) || [];
    if (columnId != null) {
      newLayer = deleteColumn({
        layer: tempLayer,
        columnId,
        indexPattern
      });
    }
  }
  if (previousDefinition.input === 'fullReference') {
    previousColumn.references.forEach(id => {
      newLayer = deleteColumn({
        layer: tempLayer,
        columnId: id,
        indexPattern
      });
    });
  }
  return newLayer;
}
function canTransition({
  layer,
  columnId,
  op,
  field,
  indexPattern,
  filterOperations,
  visualizationGroups,
  dateRange
}) {
  const previousColumn = layer.columns[columnId];
  if (!previousColumn) {
    return true;
  }
  if (previousColumn.operationType === op) {
    return true;
  }
  try {
    var _newLayer$incompleteC, _newDefinition$getErr, _newDefinition$getErr2;
    const newLayer = replaceColumn({
      layer,
      columnId,
      op,
      field,
      indexPattern,
      visualizationGroups
    });
    const newDefinition = _definitions.operationDefinitionMap[op];
    const newColumn = newLayer.columns[columnId];
    return Boolean(newColumn) && !((_newLayer$incompleteC = newLayer.incompleteColumns) !== null && _newLayer$incompleteC !== void 0 && _newLayer$incompleteC[columnId]) && filterOperations(newColumn) && !((_newDefinition$getErr = newDefinition.getErrorMessage) !== null && _newDefinition$getErr !== void 0 && (_newDefinition$getErr2 = _newDefinition$getErr.call(newDefinition, newLayer, columnId, indexPattern, dateRange)) !== null && _newDefinition$getErr2 !== void 0 && _newDefinition$getErr2.length);
  } catch (e) {
    return false;
  }
}

/**
 * Function to transition to a fullReference from any different operation.
 * It is always possible to transition to a fullReference, but there are multiple
 * passes needed to copy all the previous state. These are the passes in priority
 * order, each of which has a unit test:
 *
 * 1. Case ref1: referenced columns are an exact match
 *    Side effect: Modifies the reference list directly
 * 2. Case new1: the previous column is an exact match.
 *    Side effect: Deletes and then inserts the previous column
 * 3. Case new2: the reference supports `none` inputs, like filters. not visible in the UI.
 *    Side effect: Inserts a new column
 * 4. Case new3, new4: Fuzzy matching on the previous field
 *    Side effect: Inserts a new column, or an incomplete column
 * 5. Fuzzy matching based on the previous references (case new6)
 *    Side effect: Inserts a new column, or an incomplete column
 *    Side effect: Modifies the reference list directly
 * 6. Case new6: Fall back by generating the column with empty references
 */
function applyReferenceTransition({
  layer,
  columnId,
  previousColumn,
  op,
  indexPattern,
  visualizationGroups
}) {
  const operationDefinition = _definitions.operationDefinitionMap[op];
  if (operationDefinition.input !== 'fullReference') {
    throw new Error(`Requirements for transitioning are not met`);
  }
  let hasExactMatch = false;
  let hasFieldMatch = false;
  const unusedReferencesQueue = 'references' in previousColumn ? [...previousColumn.references] : [];
  const referenceIds = operationDefinition.requiredReferences.map(validation => {
    const newId = (0, _id_generator.generateId)();

    // First priority is to use any references that can be kept (case ref1)
    if (unusedReferencesQueue.length) {
      const otherColumn = layer.columns[unusedReferencesQueue[0]];
      if (isColumnValidAsReference({
        validation,
        column: otherColumn
      })) {
        return unusedReferencesQueue.shift();
      }
    }

    // Second priority is to wrap around the previous column (case new1)
    if (!hasExactMatch && isColumnValidAsReference({
      validation,
      column: previousColumn
    })) {
      hasExactMatch = true;
      const newLayer = {
        ...layer,
        columns: {
          ...layer.columns,
          [newId]: {
            ...previousColumn,
            // drop the filter for the referenced column because the wrapping operation
            // is filterable as well and will handle it one level higher.
            filter: operationDefinition.filterable ? undefined : previousColumn.filter,
            timeShift: operationDefinition.shiftable ? undefined : previousColumn.timeShift
          }
        }
      };
      layer = updateDefaultLabels(adjustColumnReferencesForChangedColumn({
        ...newLayer,
        columnOrder: getColumnOrder(newLayer)
      }, newId), indexPattern);
      return newId;
    }

    // Look for any fieldless operations that can be inserted directly (case new2)
    if (validation.input.includes('none')) {
      const validOperations = _definitions.operationDefinitions.filter(def => {
        if (def.input !== 'none') return;
        return isOperationAllowedAsReference({
          validation,
          operationType: def.type,
          indexPattern
        });
      });
      if (validOperations.length === 1) {
        layer = insertNewColumn({
          layer,
          columnId: newId,
          op: validOperations[0].type,
          indexPattern,
          visualizationGroups
        });
        return newId;
      }
    }

    // Try to reuse the previous field by finding a possible operation. Because we've alredy
    // checked for an exact operation match, this is guaranteed to be different from previousColumn
    if (!hasFieldMatch && 'sourceField' in previousColumn && validation.input.includes('field')) {
      const previousField = indexPattern.getFieldByName(previousColumn.sourceField);
      const defIgnoringfield = _definitions.operationDefinitions.filter(def => def.input === 'field' && isOperationAllowedAsReference({
        validation,
        operationType: def.type,
        indexPattern
      })).sort((0, _operations.getSortScoreByPriorityForField)(previousField));

      // No exact match found, so let's determine that the current field can be reused
      const defWithField = defIgnoringfield.filter(def => {
        if (!previousField) return;
        return isOperationAllowedAsReference({
          validation,
          operationType: def.type,
          field: previousField,
          indexPattern
        });
      });
      if (defWithField.length > 0) {
        // Found the best match that keeps the field (case new3)
        hasFieldMatch = true;
        layer = insertNewColumn({
          layer,
          columnId: newId,
          op: defWithField[0].type,
          indexPattern,
          field: indexPattern.getFieldByName(previousColumn.sourceField),
          visualizationGroups
        });
        return newId;
      } else if (defIgnoringfield.length === 1) {
        // Can't use the field, but there is an exact match on the operation (case new4)
        hasFieldMatch = true;
        layer = {
          ...layer,
          incompleteColumns: {
            ...layer.incompleteColumns,
            [newId]: {
              operationType: defIgnoringfield[0].type
            }
          }
        };
        return newId;
      }
    }

    // Look for field-based references that we can use to assign a new field-based operation from (case new5)
    if (unusedReferencesQueue.length) {
      const otherColumn = layer.columns[unusedReferencesQueue[0]];
      if (otherColumn && 'sourceField' in otherColumn && validation.input.includes('field')) {
        const previousField = indexPattern.getFieldByName(otherColumn.sourceField);
        if (previousField) {
          const defWithField = _definitions.operationDefinitions.filter(def => def.input === 'field' && isOperationAllowedAsReference({
            validation,
            operationType: def.type,
            field: previousField,
            indexPattern
          })).sort((0, _operations.getSortScoreByPriorityForField)(previousField));
          if (defWithField.length > 0) {
            layer = insertNewColumn({
              layer,
              columnId: newId,
              op: defWithField[0].type,
              indexPattern,
              field: previousField,
              visualizationGroups
            });
            return newId;
          }
        }
      }
    }

    // The reference is too ambiguous at this point, but instead of throwing an error (case new6)
    return newId;
  });
  if (unusedReferencesQueue.length) {
    unusedReferencesQueue.forEach(id => {
      layer = deleteColumn({
        layer,
        columnId: id,
        indexPattern
      });
    });
  }
  layer = {
    ...layer,
    columns: {
      ...layer.columns,
      [columnId]: operationDefinition.buildColumn({
        indexPattern,
        layer,
        referenceIds,
        previousColumn
      })
    }
  };
  return updateDefaultLabels(adjustColumnReferencesForChangedColumn({
    ...layer,
    columnOrder: getColumnOrder(layer)
  }, columnId), indexPattern);
}
function copyCustomLabel(newColumn, previousOptions) {
  const adjustedColumn = {
    ...newColumn
  };
  const operationChanged = newColumn.operationType !== previousOptions.operationType;
  const fieldChanged = ('sourceField' in newColumn && newColumn.sourceField) !== ('sourceField' in previousOptions && previousOptions.sourceField);
  // only copy custom label if either used operation or used field stayed the same
  if (previousOptions.customLabel && (!operationChanged || !fieldChanged)) {
    adjustedColumn.customLabel = true;
    adjustedColumn.label = previousOptions.label;
  }
  return adjustedColumn;
}
function addBucket(layer, column, addedColumnId, visualizationGroups, targetGroup, respectOrder) {
  const [buckets, metrics] = (0, _lodash.partition)(layer.columnOrder, colId => layer.columns[colId].isBucketed);
  const oldDateHistogramIndex = layer.columnOrder.findIndex(columnId => layer.columns[columnId].operationType === 'date_histogram');
  let updatedColumnOrder = [];
  if (oldDateHistogramIndex > -1 && column.operationType === 'terms' && !respectOrder) {
    // Insert the new terms bucket above the first date histogram
    updatedColumnOrder = [...buckets.slice(0, oldDateHistogramIndex), addedColumnId, ...buckets.slice(oldDateHistogramIndex, buckets.length), ...metrics];
  } else {
    // Insert the new bucket after existing buckets. Users will see the same data
    // they already had, with an extra level of detail.
    updatedColumnOrder = [...buckets, addedColumnId, ...metrics];
  }
  updatedColumnOrder = reorderByGroups(visualizationGroups, updatedColumnOrder, targetGroup, addedColumnId);
  const tempLayer = {
    ...resetIncomplete(layer, addedColumnId),
    columns: {
      ...layer.columns,
      [addedColumnId]: column
    },
    columnOrder: updatedColumnOrder
  };
  return adjustColumnReferencesForChangedColumn({
    ...tempLayer,
    columnOrder: getColumnOrder(tempLayer)
  }, addedColumnId);
}
function reorderByGroups(visualizationGroups, updatedColumnOrder, targetGroup, addedColumnId) {
  var _visualizationGroups$;
  const hidesColumnGrouping = targetGroup && ((_visualizationGroups$ = visualizationGroups.find(group => group.groupId === targetGroup)) === null || _visualizationGroups$ === void 0 ? void 0 : _visualizationGroups$.hideGrouping);

  // if column grouping is disabled, keep bucket aggregations in the same order as the groups
  // if grouping is known
  if (hidesColumnGrouping) {
    const orderedVisualizationGroups = [...visualizationGroups];
    orderedVisualizationGroups.sort((group1, group2) => {
      if (typeof group1.nestingOrder === undefined) {
        return -1;
      }
      if (typeof group2.nestingOrder === undefined) {
        return 1;
      }
      return group1.nestingOrder - group2.nestingOrder;
    });
    const columnGroupIndex = {};
    updatedColumnOrder.forEach(columnId => {
      const groupIndex = orderedVisualizationGroups.findIndex(group => columnId === addedColumnId && group.groupId === targetGroup || group.accessors.some(acc => acc.columnId === columnId));
      if (groupIndex !== -1) {
        columnGroupIndex[columnId] = groupIndex;
      } else {
        // referenced columns won't show up in visualization groups - put them in the back of the list. This will work as they are always metrics
        columnGroupIndex[columnId] = updatedColumnOrder.length;
      }
    });
    return [...updatedColumnOrder].sort((a, b) => {
      return columnGroupIndex[a] - columnGroupIndex[b];
    });
  } else {
    return updatedColumnOrder;
  }
}
function addMetric(layer, column, addedColumnId) {
  const tempLayer = {
    ...resetIncomplete(layer, addedColumnId),
    columns: {
      ...layer.columns,
      [addedColumnId]: column
    }
  };
  return adjustColumnReferencesForChangedColumn({
    ...tempLayer,
    columnOrder: getColumnOrder(tempLayer)
  }, addedColumnId);
}
function getMetricOperationTypes(field) {
  return _definitions.operationDefinitions.sort((0, _operations.getSortScoreByPriorityForField)(field)).filter(definition => {
    if (definition.input !== 'field') return;
    const metadata = definition.getPossibleOperationForField(field);
    return metadata && !metadata.isBucketed && metadata.dataType === 'number';
  });
}
function updateColumnLabel({
  layer,
  columnId,
  customLabel
}) {
  const oldColumn = layer.columns[columnId];
  return {
    ...layer,
    columns: {
      ...layer.columns,
      [columnId]: {
        ...oldColumn,
        label: customLabel !== undefined ? customLabel : oldColumn.label,
        customLabel: Boolean(customLabel)
      }
    }
  };
}
function updateColumnParam({
  layer,
  columnId,
  paramName,
  value
}) {
  const currentColumn = layer.columns[columnId];
  return {
    ...layer,
    columns: {
      ...layer.columns,
      [columnId]: {
        ...currentColumn,
        params: {
          ...('params' in currentColumn ? currentColumn.params : {}),
          [paramName]: value
        }
      }
    }
  };
}
function adjustColumnReferences(layer) {
  const newColumns = {
    ...layer.columns
  };
  Object.keys(newColumns).forEach(currentColumnId => {
    const currentColumn = newColumns[currentColumnId];
    if (currentColumn !== null && currentColumn !== void 0 && currentColumn.operationType) {
      const operationDefinition = _definitions.operationDefinitionMap[currentColumn.operationType];
      newColumns[currentColumnId] = operationDefinition.onOtherColumnChanged ? operationDefinition.onOtherColumnChanged({
        ...layer,
        columns: newColumns
      }, currentColumnId) : currentColumn;
    }
  });
  return {
    ...layer,
    columns: newColumns
  };
}
function adjustColumnReferencesForChangedColumn(layer, changedColumnId) {
  const newColumns = {
    ...layer.columns
  };
  Object.keys(newColumns).forEach(currentColumnId => {
    if (currentColumnId !== changedColumnId) {
      const currentColumn = newColumns[currentColumnId];
      const operationDefinition = _definitions.operationDefinitionMap[currentColumn.operationType];
      newColumns[currentColumnId] = operationDefinition.onOtherColumnChanged ? operationDefinition.onOtherColumnChanged({
        ...layer,
        columns: newColumns
      }, currentColumnId) : currentColumn;
    }
  });
  return {
    ...layer,
    columns: newColumns
  };
}
function deleteColumn({
  layer,
  columnId,
  indexPattern
}) {
  const column = layer.columns[columnId];
  if (!column) {
    const newIncomplete = {
      ...(layer.incompleteColumns || {})
    };
    delete newIncomplete[columnId];
    return {
      ...layer,
      columnOrder: layer.columnOrder.filter(id => id !== columnId),
      incompleteColumns: newIncomplete
    };
  }
  const extraDeletions = 'references' in column ? column.references : [];
  const hypotheticalColumns = {
    ...layer.columns
  };
  delete hypotheticalColumns[columnId];
  let newLayer = adjustColumnReferencesForChangedColumn({
    ...layer,
    columns: hypotheticalColumns
  }, columnId);
  extraDeletions.forEach(id => {
    newLayer = deleteColumn({
      layer: newLayer,
      columnId: id,
      indexPattern
    });
  });
  const newIncomplete = {
    ...(newLayer.incompleteColumns || {})
  };
  delete newIncomplete[columnId];
  return updateDefaultLabels({
    ...newLayer,
    columnOrder: getColumnOrder(newLayer),
    incompleteColumns: newIncomplete
  }, indexPattern);
}

// Column order mostly affects the visual order in the UI. It is derived
// from the columns objects, respecting any existing columnOrder relationships,
// but allowing new columns to be inserted
//
// This does NOT topologically sort references, as this would cause the order in the UI
// to change. Reference order is determined before creating the pipeline in to_expression
function getColumnOrder(layer) {
  const entries = Object.entries(layer.columns);
  entries.sort(([idA], [idB]) => {
    const indexA = layer.columnOrder.indexOf(idA);
    const indexB = layer.columnOrder.indexOf(idB);
    if (indexA > -1 && indexB > -1) {
      return indexA - indexB;
    } else if (indexA > -1) {
      return -1;
    } else {
      return 1;
    }
  });
  const [aggregations, metrics] = (0, _lodash.partition)(entries, ([, col]) => col.isBucketed);
  return aggregations.map(([id]) => id).concat(metrics.map(([id]) => id));
}

// Splits existing columnOrder into the three categories
function getExistingColumnGroups(layer) {
  const [direct, referenced] = (0, _lodash.partition)(layer.columnOrder, columnId => layer.columns[columnId] && !('references' in layer.columns[columnId]));
  return [...(0, _lodash.partition)(direct, columnId => {
    var _layer$columns$column;
    return (_layer$columns$column = layer.columns[columnId]) === null || _layer$columns$column === void 0 ? void 0 : _layer$columns$column.isBucketed;
  }), referenced];
}

/**
 * Returns true if the given column can be applied to the given index pattern
 */
function isColumnTransferable(column, newIndexPattern, layer) {
  return _definitions.operationDefinitionMap[column.operationType].isTransferable(column, newIndexPattern, _definitions.operationDefinitionMap) && (!('references' in column) || column.references.every(columnId => isColumnTransferable(layer.columns[columnId], newIndexPattern, layer)));
}
function updateLayerIndexPattern(layer, newIndexPattern) {
  const keptColumns = (0, _lodash.pickBy)(layer.columns, column => {
    return isColumnTransferable(column, newIndexPattern, layer);
  });
  const newColumns = (0, _lodash.mapValues)(keptColumns, column => {
    const operationDefinition = _definitions.operationDefinitionMap[column.operationType];
    return operationDefinition.transfer ? operationDefinition.transfer(column, newIndexPattern) : column;
  });
  const newColumnOrder = layer.columnOrder.filter(columnId => newColumns[columnId]);
  return {
    ...layer,
    indexPatternId: newIndexPattern.id,
    columns: newColumns,
    columnOrder: newColumnOrder
  };
}
/**
 * Collects all errors from the columns in the layer, for display in the workspace. This includes:
 *
 * - All columns have complete references
 * - All column references are valid
 * - All prerequisites are met
 * - If timeshift is used, terms go before date histogram
 * - If timeshift is used, only a single date histogram can be used
 */
function getErrorMessages(layer, indexPattern, state, layerId, core, data) {
  const columns = Object.entries(layer.columns);
  const visibleManagedReferences = columns.filter(([columnId, column]) => !isReferenced(layer, columnId) && _definitions.operationDefinitionMap[column.operationType].input === 'managedReference');
  const skippedColumns = visibleManagedReferences.flatMap(([columnId]) => getManagedColumnsFrom(columnId, layer.columns).map(([id]) => id));
  const errors = columns.flatMap(([columnId, column]) => {
    if (skippedColumns.includes(columnId)) {
      return;
    }
    const def = _definitions.operationDefinitionMap[column.operationType];
    if (def.getErrorMessage) {
      const currentTimeRange = data.query.timefilter.timefilter.getAbsoluteTime();
      return def.getErrorMessage(layer, columnId, indexPattern, {
        fromDate: currentTimeRange.from,
        toDate: currentTimeRange.to
      }, _definitions.operationDefinitionMap, core.uiSettings.get(_public.UI_SETTINGS.HISTOGRAM_BAR_TARGET));
    }
  }).map(errorMessage => {
    if (typeof errorMessage !== 'object') {
      return errorMessage;
    }
    return {
      ...errorMessage,
      fixAction: errorMessage.fixAction ? {
        ...errorMessage.fixAction,
        newState: async frame => ({
          ...state,
          layers: {
            ...state.layers,
            [layerId]: await errorMessage.fixAction.newState(data, core, frame, layerId)
          }
        })
      } : undefined
    };
  })
  // remove the undefined values
  .filter(v => v != null);
  return errors.length ? errors : undefined;
}
function isReferenced(layer, columnId) {
  const allReferences = Object.values(layer.columns).flatMap(col => 'references' in col ? col.references : []);
  return allReferences.includes(columnId);
}
const computeReferenceLookup = (0, _memoizeOne.default)(layer => {
  // speed up things for deep chains as in formula
  const refLookup = {};
  for (const [parentId, col] of Object.entries(layer.columns)) {
    if ('references' in col) {
      for (const colId of col.references) {
        refLookup[colId] = parentId;
      }
    }
  }
  return refLookup;
});

/**
 * Given a columnId, returns the visible root column id for it
 * This is useful to map internal properties of referenced columns to the visible column
 * @param layer
 * @param columnId
 * @returns id of the reference root
 */
function getReferenceRoot(layer, columnId) {
  const refLookup = computeReferenceLookup(layer);
  let currentId = columnId;
  while (isReferenced(layer, currentId)) {
    currentId = refLookup[currentId];
  }
  return currentId;
}
function getReferencedColumnIds(layer, columnId) {
  const referencedIds = [];
  function collect(id) {
    const column = layer.columns[id];
    if (column && 'references' in column) {
      const columnReferences = column.references;
      // only record references which have created columns yet
      const existingReferences = columnReferences.filter(reference => Boolean(layer.columns[reference]));
      referencedIds.push(...existingReferences);
      existingReferences.forEach(collect);
    }
  }
  collect(columnId);
  return referencedIds;
}
function hasTermsWithManyBuckets(layer) {
  return layer.columnOrder.some(columnId => {
    const column = layer.columns[columnId];
    if (column) {
      return (0, _helpers.isColumnOfType)('terms', column) && column.params.size > 5;
    }
  });
}
function isOperationAllowedAsReference({
  operationType,
  validation,
  field,
  indexPattern
}) {
  const operationDefinition = _definitions.operationDefinitionMap[operationType];
  let hasValidMetadata = true;
  if (field && operationDefinition.input === 'field') {
    const metadata = operationDefinition.getPossibleOperationForField(field);
    hasValidMetadata = Boolean(metadata) && validation.validateMetadata(metadata, operationType, field.name);
  } else if (operationDefinition.input === 'none') {
    const metadata = operationDefinition.getPossibleOperation(indexPattern);
    hasValidMetadata = Boolean(metadata) && validation.validateMetadata(metadata, operationType);
  } else if (operationDefinition.input === 'fullReference') {
    const metadata = operationDefinition.getPossibleOperation(indexPattern);
    hasValidMetadata = Boolean(metadata) && validation.validateMetadata(metadata, operationType);
  } else {
    // TODO: How can we validate the metadata without a specific field?
  }
  return validation.input.includes(operationDefinition.input) && (!validation.specificOperations || validation.specificOperations.includes(operationType)) && hasValidMetadata;
}

// Labels need to be updated when columns are added because reference-based column labels
// are sometimes copied into the parents
function updateDefaultLabels(layer, indexPattern) {
  const copiedColumns = {
    ...layer.columns
  };
  layer.columnOrder.forEach(id => {
    const col = copiedColumns[id];
    if (!col.customLabel) {
      copiedColumns[id] = {
        ...col,
        label: _definitions.operationDefinitionMap[col.operationType].getDefaultLabel(col, copiedColumns, indexPattern)
      };
    }
  });
  return {
    ...layer,
    columns: copiedColumns
  };
}
function resetIncomplete(layer, columnId) {
  var _layer$incompleteColu4;
  const incompleteColumns = {
    ...((_layer$incompleteColu4 = layer.incompleteColumns) !== null && _layer$incompleteColu4 !== void 0 ? _layer$incompleteColu4 : {})
  };
  delete incompleteColumns[columnId];
  return {
    ...layer,
    incompleteColumns
  };
}

// managedReferences have a relaxed policy about operation allowed, so let them pass
function maybeValidateOperations({
  column,
  validation
}) {
  if (!validation.specificOperations) {
    return true;
  }
  if (_definitions.operationDefinitionMap[column.operationType].input === 'managedReference') {
    return true;
  }
  return validation.specificOperations.includes(column.operationType);
}
function isColumnValidAsReference({
  column,
  validation
}) {
  if (!column) return false;
  const operationType = column.operationType;
  const operationDefinition = _definitions.operationDefinitionMap[operationType];
  if (!operationDefinition) {
    throw new Error('No suitable operation definition found for ' + operationType);
  }
  return validation.input.includes(operationDefinition.input) && maybeValidateOperations({
    column,
    validation
  }) && validation.validateMetadata(column, operationType, 'sourceField' in column ? column.sourceField : undefined);
}
function getManagedColumnsFrom(columnId, columns) {
  const allNodes = {};
  Object.entries(columns).forEach(([id, col]) => {
    allNodes[id] = 'references' in col ? [...col.references] : [];
  });
  const queue = allNodes[columnId];
  const store = [];
  while (queue.length > 0) {
    const nextId = queue.shift();
    store.push([nextId, columns[nextId]]);
    queue.push(...allNodes[nextId]);
  }
  return store.filter(([, column]) => column);
}