"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ActionsClientGeminiChatModel = void 0;
exports.convertAuthorToRole = convertAuthorToRole;
exports.convertBaseMessagesToContent = convertBaseMessagesToContent;
exports.convertMessageContentToParts = convertMessageContentToParts;
exports.convertResponseContentToChatGenerationChunk = convertResponseContentToChatGenerationChunk;
exports.getMessageAuthor = getMessageAuthor;
var _messages = require("@langchain/core/messages");
var _outputs = require("@langchain/core/outputs");
var _googleGenai = require("@langchain/google-genai");
var _fp = require("lodash/fp");
function _classPrivateFieldInitSpec(e, t, a) { _checkPrivateRedeclaration(e, t), t.set(e, a); }
function _checkPrivateRedeclaration(e, t) { if (t.has(e)) throw new TypeError("Cannot initialize the same private elements twice on an object"); }
function _classPrivateFieldGet(s, a) { return s.get(_assertClassBrand(s, a)); }
function _classPrivateFieldSet(s, a, r) { return s.set(_assertClassBrand(s, a), r), r; }
function _assertClassBrand(e, t, n) { if ("function" == typeof e ? e === t : e.has(t)) return arguments.length < 3 ? t : n; throw new TypeError("Private element is not present on this object"); } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const DEFAULT_GEMINI_TEMPERATURE = 0;
var _actionsClient = /*#__PURE__*/new WeakMap();
var _connectorId = /*#__PURE__*/new WeakMap();
var _temperature = /*#__PURE__*/new WeakMap();
var _model = /*#__PURE__*/new WeakMap();
class ActionsClientGeminiChatModel extends _googleGenai.ChatGoogleGenerativeAI {
  constructor({
    actionsClient,
    connectorId,
    ...props
  }) {
    var _props$maxTokens, _props$model, _props$temperature;
    super({
      ...props,
      apiKey: 'asda',
      maxOutputTokens: (_props$maxTokens = props.maxTokens) !== null && _props$maxTokens !== void 0 ? _props$maxTokens : 2048
    });
    // LangChain needs model to be defined for logging purposes
    _classPrivateFieldInitSpec(this, _actionsClient, void 0);
    _classPrivateFieldInitSpec(this, _connectorId, void 0);
    _classPrivateFieldInitSpec(this, _temperature, void 0);
    _classPrivateFieldInitSpec(this, _model, void 0);
    this.model = (_props$model = props.model) !== null && _props$model !== void 0 ? _props$model : this.model;
    // If model is not specified by consumer, the connector will defin eit so do not pass
    // a LangChain default to the actionsClient
    _classPrivateFieldSet(_model, this, props.model);
    _classPrivateFieldSet(_temperature, this, (_props$temperature = props.temperature) !== null && _props$temperature !== void 0 ? _props$temperature : DEFAULT_GEMINI_TEMPERATURE);
    _classPrivateFieldSet(_actionsClient, this, actionsClient);
    _classPrivateFieldSet(_connectorId, this, connectorId);
  }
  async completionWithRetry(request, options) {
    return this.caller.callWithOptions({
      signal: options === null || options === void 0 ? void 0 : options.signal
    }, async () => {
      try {
        const requestBody = {
          actionId: _classPrivateFieldGet(_connectorId, this),
          params: {
            subAction: 'invokeAIRaw',
            subActionParams: {
              model: _classPrivateFieldGet(_model, this),
              messages: request.contents,
              tools: request.tools,
              temperature: _classPrivateFieldGet(_temperature, this)
            }
          }
        };
        const actionResult = await _classPrivateFieldGet(_actionsClient, this).execute(requestBody);
        if (actionResult.status === 'error') {
          throw new Error(`ActionsClientGeminiChatModel: action result status is error: ${actionResult === null || actionResult === void 0 ? void 0 : actionResult.message} - ${actionResult === null || actionResult === void 0 ? void 0 : actionResult.serviceMessage}`);
        }
        return {
          response: {
            ...actionResult.data,
            functionCalls: () => {
              var _actionResult$data, _actionResult$data$ca, _actionResult$data$ca2, _actionResult$data$ca3, _actionResult$data2, _actionResult$data2$c, _actionResult$data2$c2;
              return (_actionResult$data = actionResult.data) !== null && _actionResult$data !== void 0 && (_actionResult$data$ca = _actionResult$data.candidates) !== null && _actionResult$data$ca !== void 0 && (_actionResult$data$ca2 = _actionResult$data$ca[0]) !== null && _actionResult$data$ca2 !== void 0 && (_actionResult$data$ca3 = _actionResult$data$ca2.content) !== null && _actionResult$data$ca3 !== void 0 && _actionResult$data$ca3.parts[0].functionCall ? [(_actionResult$data2 = actionResult.data) === null || _actionResult$data2 === void 0 ? void 0 : (_actionResult$data2$c = _actionResult$data2.candidates) === null || _actionResult$data2$c === void 0 ? void 0 : (_actionResult$data2$c2 = _actionResult$data2$c[0]) === null || _actionResult$data2$c2 === void 0 ? void 0 : _actionResult$data2$c2.content.parts[0].functionCall] : [];
            }
          }
        };
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
      } catch (e) {
        var _e$message;
        // TODO: Improve error handling
        if ((_e$message = e.message) !== null && _e$message !== void 0 && _e$message.includes('400 Bad Request')) {
          e.status = 400;
        }
        throw e;
      }
    });
  }
  async *_streamResponseChunks(messages, options, runManager) {
    const prompt = convertBaseMessagesToContent(messages, this._isMultimodalModel);
    const parameters = this.invocationParams(options);
    const request = {
      ...parameters,
      contents: prompt
    };
    const stream = await this.caller.callWithOptions({
      signal: options === null || options === void 0 ? void 0 : options.signal
    }, async () => {
      const requestBody = {
        actionId: _classPrivateFieldGet(_connectorId, this),
        params: {
          subAction: 'invokeStream',
          subActionParams: {
            model: _classPrivateFieldGet(_model, this),
            messages: request.contents.reduce((acc, item) => {
              if (!(acc !== null && acc !== void 0 && acc.length)) {
                acc.push(item);
                return acc;
              }
              if (acc[acc.length - 1].role === item.role) {
                acc[acc.length - 1].parts = acc[acc.length - 1].parts.concat(item.parts);
                return acc;
              }
              acc.push(item);
              return acc;
            }, []),
            temperature: _classPrivateFieldGet(_temperature, this),
            tools: request.tools
          }
        }
      };
      const actionResult = await _classPrivateFieldGet(_actionsClient, this).execute(requestBody);
      if (actionResult.status === 'error') {
        throw new Error(`ActionsClientGeminiChatModel: action result status is error: ${actionResult === null || actionResult === void 0 ? void 0 : actionResult.message} - ${actionResult === null || actionResult === void 0 ? void 0 : actionResult.serviceMessage}`);
      }
      const readable = (0, _fp.get)('data', actionResult);
      if (typeof (readable === null || readable === void 0 ? void 0 : readable.read) !== 'function') {
        throw new Error('Action result status is error: result is not streamable');
      }
      return readable;
    });
    let usageMetadata;
    let index = 0;
    let partialStreamChunk = '';
    for await (const rawStreamChunk of stream) {
      var _parsedStreamChunk$ca, _parsedStreamChunk$ca2;
      const streamChunk = rawStreamChunk.toString();
      const nextChunk = `${partialStreamChunk + streamChunk}`;
      let parsedStreamChunk = null;
      try {
        parsedStreamChunk = JSON.parse(nextChunk.replaceAll('data: ', '').replaceAll('\r\n', ''));
        partialStreamChunk = '';
      } catch (_) {
        partialStreamChunk += nextChunk;
      }
      if (parsedStreamChunk !== null && !((_parsedStreamChunk$ca = parsedStreamChunk.candidates) !== null && _parsedStreamChunk$ca !== void 0 && (_parsedStreamChunk$ca2 = _parsedStreamChunk$ca[0]) !== null && _parsedStreamChunk$ca2 !== void 0 && _parsedStreamChunk$ca2.finishReason)) {
        const response = {
          ...parsedStreamChunk,
          functionCalls: () => {
            var _parsedStreamChunk, _parsedStreamChunk$ca3, _parsedStreamChunk$ca4, _parsedStreamChunk$ca5, _parsedStreamChunk$ca6;
            return (_parsedStreamChunk = parsedStreamChunk) !== null && _parsedStreamChunk !== void 0 && (_parsedStreamChunk$ca3 = _parsedStreamChunk.candidates) !== null && _parsedStreamChunk$ca3 !== void 0 && (_parsedStreamChunk$ca4 = _parsedStreamChunk$ca3[0]) !== null && _parsedStreamChunk$ca4 !== void 0 && _parsedStreamChunk$ca4.content.parts[0].functionCall ? [(_parsedStreamChunk$ca5 = parsedStreamChunk.candidates) === null || _parsedStreamChunk$ca5 === void 0 ? void 0 : (_parsedStreamChunk$ca6 = _parsedStreamChunk$ca5[0]) === null || _parsedStreamChunk$ca6 === void 0 ? void 0 : _parsedStreamChunk$ca6.content.parts[0].functionCall] : [];
          }
        };
        if ('usageMetadata' in response && this.streamUsage !== false && options.streamUsage !== false) {
          const genAIUsageMetadata = response.usageMetadata;
          if (!usageMetadata) {
            usageMetadata = {
              input_tokens: genAIUsageMetadata.promptTokenCount,
              output_tokens: genAIUsageMetadata.candidatesTokenCount,
              total_tokens: genAIUsageMetadata.totalTokenCount
            };
          } else {
            // Under the hood, LangChain combines the prompt tokens. Google returns the updated
            // total each time, so we need to find the difference between the tokens.
            const outputTokenDiff = genAIUsageMetadata.candidatesTokenCount - usageMetadata.output_tokens;
            usageMetadata = {
              input_tokens: 0,
              output_tokens: outputTokenDiff,
              total_tokens: outputTokenDiff
            };
          }
        }
        const chunk = convertResponseContentToChatGenerationChunk(response, {
          usageMetadata,
          index
        });
        index += 1;
        if (chunk) {
          var _chunk$text;
          yield chunk;
          await (runManager === null || runManager === void 0 ? void 0 : runManager.handleLLMNewToken((_chunk$text = chunk.text) !== null && _chunk$text !== void 0 ? _chunk$text : ''));
        }
      }
    }
  }
}
exports.ActionsClientGeminiChatModel = ActionsClientGeminiChatModel;
function convertResponseContentToChatGenerationChunk(response, extra) {
  var _content$parts$0$text, _content$parts$;
  if (!response.candidates || response.candidates.length === 0) {
    return null;
  }
  const functionCalls = response.functionCalls();
  const [candidate] = response.candidates;
  const {
    content,
    ...generationInfo
  } = candidate;
  const text = (_content$parts$0$text = content === null || content === void 0 ? void 0 : (_content$parts$ = content.parts[0]) === null || _content$parts$ === void 0 ? void 0 : _content$parts$.text) !== null && _content$parts$0$text !== void 0 ? _content$parts$0$text : '';
  const toolCallChunks = [];
  if (functionCalls) {
    toolCallChunks.push(...functionCalls.map(fc => ({
      ...fc,
      args: JSON.stringify(fc.args),
      index: extra.index,
      type: 'tool_call_chunk'
    })));
  }
  return new _outputs.ChatGenerationChunk({
    text,
    message: new _messages.AIMessageChunk({
      content: text,
      name: !content ? undefined : content.role,
      tool_call_chunks: toolCallChunks,
      // Each chunk can have unique "generationInfo", and merging strategy is unclear,
      // so leave blank for now.
      additional_kwargs: {},
      usage_metadata: extra.usageMetadata
    }),
    generationInfo
  });
}
function convertAuthorToRole(author) {
  switch (author) {
    /**
     *  Note: Gemini currently is not supporting system messages
     *  we will convert them to human messages and merge with following
     * */
    case 'ai':
    case 'model':
      // getMessageAuthor returns message.name. code ex.: return message.name ?? type;
      return 'model';
    case 'system':
    case 'human':
      return 'user';
    case 'tool':
    case 'function':
      return 'function';
    default:
      throw new Error(`Unknown / unsupported author: ${author}`);
  }
}
function convertBaseMessagesToContent(messages, isMultimodalModel) {
  return messages.reduce((acc, message, index) => {
    if (!(0, _messages.isBaseMessage)(message)) {
      throw new Error('Unsupported message input');
    }
    const author = getMessageAuthor(message);
    if (author === 'system' && index !== 0) {
      throw new Error('System message should be the first one');
    }
    const role = convertAuthorToRole(author);
    const parts = convertMessageContentToParts(message, isMultimodalModel);
    if (acc.mergeWithPreviousContent) {
      const prevContent = acc.content[acc.content.length - 1];
      if (!prevContent) {
        throw new Error('There was a problem parsing your system message. Please try a prompt without one.');
      }
      prevContent.parts.push(...parts);
      return {
        mergeWithPreviousContent: false,
        content: acc.content
      };
    }
    let actualRole = role;
    if (actualRole === 'function') {
      // GenerativeAI API will throw an error if the role is not "user" or "model."
      actualRole = 'user';
    }
    const content = {
      role: actualRole,
      parts
    };
    return {
      mergeWithPreviousContent: author === 'system',
      content: [...acc.content, content]
    };
  }, {
    content: [],
    mergeWithPreviousContent: false
  }).content;
}
function convertMessageContentToParts(message, isMultimodalModel) {
  if (typeof message.content === 'string' && message.content !== '') {
    return [{
      text: message.content
    }];
  }
  let functionCalls = [];
  let functionResponses = [];
  let messageParts = [];
  if ('tool_calls' in message && Array.isArray(message.tool_calls) && message.tool_calls.length > 0) {
    functionCalls = message.tool_calls.map(tc => ({
      functionCall: {
        name: tc.name,
        args: tc.args
      }
    }));
  } else if (message._getType() === 'tool' && message.name && message.content) {
    functionResponses = [{
      functionResponse: {
        name: message.name,
        response: message.content
      }
    }];
  } else if (Array.isArray(message.content)) {
    messageParts = message.content.map(c => {
      if (c.type === 'text') {
        return {
          text: c.text
        };
      }
      if (c.type === 'image_url') {
        if (!isMultimodalModel) {
          throw new Error(`This model does not support images`);
        }
        let source;
        if (typeof c.image_url === 'string') {
          source = c.image_url;
        } else if (typeof c.image_url === 'object' && 'url' in c.image_url) {
          source = c.image_url.url;
        } else {
          throw new Error('Please provide image as base64 encoded data URL');
        }
        const [dm, data] = source.split(',');
        if (!dm.startsWith('data:')) {
          throw new Error('Please provide image as base64 encoded data URL');
        }
        const [mimeType, encoding] = dm.replace(/^data:/, '').split(';');
        if (encoding !== 'base64') {
          throw new Error('Please provide image as base64 encoded data URL');
        }
        return {
          inlineData: {
            data,
            mimeType
          }
        };
      } else if (c.type === 'media') {
        return messageContentMedia(c);
      } else if (c.type === 'tool_use') {
        return {
          functionCall: {
            name: c.name,
            args: c.input
          }
        };
      }
      throw new Error(`Unknown content type ${c.type}`);
    });
  }
  return [...messageParts, ...functionCalls, ...functionResponses];
}
function getMessageAuthor(message) {
  var _message$name;
  const type = message._getType();
  if (_messages.ChatMessage.isInstance(message)) {
    return message.role;
  }
  if (type === 'tool') {
    return type;
  }
  return (_message$name = message.name) !== null && _message$name !== void 0 ? _message$name : type;
}

// will be removed once FileDataPart is supported in @langchain/google-genai
function messageContentMedia(content) {
  if ('mimeType' in content && 'data' in content) {
    return {
      inlineData: {
        mimeType: content.mimeType,
        data: content.data
      }
    };
  }
  throw new Error('Invalid media content');
}