"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResizeChecker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _events = require("events");
var _lodash = require("lodash");
var _resizeObserverPolyfill = _interopRequireDefault(require("resize-observer-polyfill"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getSize(el) {
  return [el.clientWidth, el.clientHeight];
}

/**
 *  ResizeChecker receives an element and emits a "resize" event every time it changes size.
 */
class ResizeChecker extends _events.EventEmitter {
  constructor(el, args = {}) {
    super();
    (0, _defineProperty2.default)(this, "destroyed", false);
    (0, _defineProperty2.default)(this, "el", void 0);
    (0, _defineProperty2.default)(this, "observer", void 0);
    (0, _defineProperty2.default)(this, "expectedSize", null);
    this.el = el;
    this.observer = new _resizeObserverPolyfill.default(() => {
      if (this.expectedSize) {
        const sameSize = (0, _lodash.isEqual)(getSize(el), this.expectedSize);
        this.expectedSize = null;
        if (sameSize) {
          // don't trigger resize notification if the size is what we expect
          return;
        }
      }
      this.emit('resize');
    });

    // Only enable the checker immediately if args.disabled wasn't set to true
    if (!args.disabled) {
      this.enable();
    }
  }
  enable() {
    if (this.destroyed) {
      // Don't allow enabling an already destroyed resize checker
      return;
    }
    // the width and height of the element that we expect to see
    // on the next resize notification. If it matches the size at
    // the time of starting observing then it we will be ignored.
    // We know that observer and el are not null since we are not yet destroyed.
    this.expectedSize = getSize(this.el);
    this.observer.observe(this.el);
  }

  /**
   *  Run a function and ignore all resizes that occur
   *  while it's running.
   */
  modifySizeWithoutTriggeringResize(block) {
    try {
      block();
    } finally {
      if (this.el) {
        this.expectedSize = getSize(this.el);
      }
    }
  }

  /**
   * Tell the ResizeChecker to shutdown, stop listenings, and never
   * emit another resize event.
   *
   * Cleans up it's listeners and timers.
   */
  destroy() {
    if (this.destroyed) {
      return;
    }
    this.destroyed = true;
    this.observer.disconnect();
    this.observer = null;
    this.expectedSize = null;
    this.el = null;
    this.removeAllListeners();
  }
}
exports.ResizeChecker = ResizeChecker;